package distrepo

import (
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"path"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/hector/internal/dist"
	"a.yandex-team.ru/security/hector/internal/procutils"
	"a.yandex-team.ru/security/hector/internal/remote"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type DistRepo struct {
	pkg       dist.Package
	fullName  string
	reference string
	debPath   string
}

var httpClient = yahttp.NewClient(yahttp.Config{
	RedirectPolicy: yahttp.RedirectNoFollow,
	DialTimeout:    2 * time.Second,
	Timeout:        120 * time.Second,
})

func NewDistRepo(pkg dist.Package) *DistRepo {
	return &DistRepo{
		pkg: pkg,
	}
}

func (r *DistRepo) Name() string {
	if r.fullName != "" {
		return r.fullName
	}
	r.fullName = r.pkg.FullName()
	return r.fullName
}

func (r *DistRepo) Parent() string {
	return ""
}

func (r *DistRepo) ProjectURL() string {
	return r.pkg.URL
}

func (r *DistRepo) CloneURL() string {
	return r.pkg.URL
}

func (r *DistRepo) PathToURL(relativePath string, lineNo int) string {
	return fmt.Sprintf("%s#%s:%d", r.pkg.URL, relativePath, lineNo)
}

func (r *DistRepo) Checkout(baseDir string, shallow bool) (skip bool, err error) {
	skip, err = r.download(baseDir)
	if skip || err != nil {
		return
	}
	if err = r.unpack(baseDir); err != nil {
		return
	}
	return
}

func (r *DistRepo) Author(relativePath string, lineNo int) (author string, err error) {
	return r.pkg.Maintainer, nil
}

func (r *DistRepo) Owners() (owners []string, err error) {
	return []string{r.pkg.Maintainer}, nil
}

func (r *DistRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *DistRepo) GenLocalPath(workDir string) string {
	return path.Join(workDir, r.Name())
}

func (r *DistRepo) DebPath(workDir string) string {
	if r.debPath != "" {
		return r.debPath
	}
	r.debPath = path.Join(workDir, path.Base(r.CloneURL()))
	return r.debPath
}

func (r *DistRepo) Reference() string {
	return r.pkg.Version
}

func (r *DistRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.Name(),
		Parent:     "",
		CloneURL:   r.CloneURL(),
		ProjectURL: r.ProjectURL(),
		Owners:     owners,
		Private:    false,
	}
}

func (r *DistRepo) download(baseDir string) (skip bool, err error) {
	notFound := false

	cloneURL := r.CloneURL()
	target := r.DebPath(baseDir)
	work := func() error {
		resp, err := httpClient.Get(cloneURL)
		if err != nil {
			return fmt.Errorf("failed to get pkg: %s", err.Error())
		}
		defer yahttp.GracefulClose(resp.Body)

		if resp.StatusCode == 404 {
			notFound = true
			// Skip not found pastes
			return nil
		}

		if resp.StatusCode != 200 {
			return fmt.Errorf("failed to get pkg, status code: %s", resp.Status)
		}

		_ = os.MkdirAll(baseDir, 0755)
		out, err := os.Create(target)
		if err != nil {
			return fmt.Errorf("failed to create resulting pkg file: %s", err.Error())
		}
		defer func() { _ = out.Close() }()

		_, err = io.Copy(out, resp.Body)
		if err != nil {
			return fmt.Errorf("failed to pkg paste: %s", err.Error())
		}
		return nil
	}

	// TODO(buglloc): enable retries back?
	if err = work(); err != nil {
		return
	}

	if notFound {
		err = errors.New("pkg not found")
		skip = true
		return
	}
	return
}

func (r *DistRepo) unpack(baseDir string) error {
	cmd := exec.Command("ar", "x", r.DebPath(baseDir))
	cmd.Dir = baseDir
	res, err := procutils.RunCmd(cmd)
	if err != nil {
		return xerrors.Errorf("failed to unpack: %w", err)
	}

	if res.ExitCode != 0 {
		return xerrors.Errorf("failed to unpack (exit code = %d): %s", res.ExitCode, res.Stderr)
	}

	files, err := ioutil.ReadDir(baseDir)
	if err != nil {
		return xerrors.Errorf("failed to list target directory: %w", err)
	}

	var toUnpack []string
	var toCleanup []string
	for _, f := range files {
		fName := f.Name()
		fPath := path.Join(baseDir, fName)
		toCleanup = append(toCleanup, fPath)
		if strings.HasPrefix(fName, "data.tar") {
			toUnpack = append(toUnpack, fPath)
		}
	}

	defer cleanup(toCleanup)
	for _, target := range toUnpack {
		cmd := exec.Command("tar", "xf", target)
		cmd.Dir = baseDir
		res, err := procutils.RunCmd(cmd)
		if err != nil {
			simplelog.Error("failed to unpack", "path", target, "err", err.Error())
			continue
		}

		if res.ExitCode != 0 {
			simplelog.Error("failed to unpack", "path", target, "stderr", res.Stderr)
			continue
		}
	}
	return nil
}

func cleanup(paths []string) {
	for _, target := range paths {
		if err := os.RemoveAll(target); err != nil {
			simplelog.Error("failed to cleanup", "path", target, "err", err.Error())
		}
	}
}
