package gistrepo

import (
	"fmt"
	"path"
	"strings"

	"github.com/google/go-github/v35/github"

	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote"
)

type GistRepo struct {
	gistRepo   *github.Gist
	cloned     bool
	local      string
	name       string
	projectURL string
	cloneURL   string
	reference  string
}

func NewGistRepo(gistRepo *github.Gist) *GistRepo {
	return &GistRepo{
		gistRepo: gistRepo,
	}
}

func (r *GistRepo) Name() string {
	if r.name == "" {
		r.name = r.gistRepo.GetID()
	}
	return r.name
}

func (r *GistRepo) Parent() string {
	return ""
}

func (r *GistRepo) ProjectURL() string {
	if r.projectURL == "" {
		r.projectURL = r.gistRepo.GetHTMLURL()
	}
	return r.projectURL
}

func (r *GistRepo) CloneURL() string {
	if r.cloneURL == "" {
		r.cloneURL = r.gistRepo.GetGitPullURL()
	}
	return r.cloneURL
}

func (r *GistRepo) PathToURL(relativePath string, lineNo int) string {
	fileName := strings.ToLower(strings.Trim(relativePath, "/"))
	fileName = strings.Replace(fileName, ".", "-", -1)
	return fmt.Sprintf("%s#file-%s-L%d",
		r.ProjectURL(), fileName, lineNo)
}

func (r *GistRepo) Checkout(target string, shallow bool) (skip bool, err error) {
	if err = gitutils.Checkout(r.CloneURL(), target, shallow); err != nil {
		return
	}

	r.cloned = true
	r.local = target
	r.reference = gitutils.GetReference(r.local)
	return
}

func (r *GistRepo) Author(relativePath string, lineNo int) (author string, err error) {
	owners, err := r.Owners()
	if err == nil && len(owners) > 0 {
		author = owners[0]
	}
	return
}

func (r *GistRepo) Owners() (owners []string, err error) {
	owners = []string{r.gistRepo.GetOwner().GetLogin()}
	return
}

func (r *GistRepo) Reference() string {
	if r.reference == "" {
		r.reference = gitutils.GetReference(r.local)
	}
	return r.reference
}

func (r *GistRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *GistRepo) GenLocalPath(workDir string) string {
	return path.Join(workDir, r.Name())
}

func (r *GistRepo) IsPrivate() bool {
	return !r.gistRepo.GetPublic()
}

func (r *GistRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.name,
		CloneURL:   r.cloneURL,
		ProjectURL: r.projectURL,
		Owners:     owners,
		Private:    r.IsPrivate(),
	}
}
