package githubrepo

import (
	"errors"
	"fmt"
	"path"
	"strings"

	"github.com/google/go-github/v35/github"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote"
)

type GithubRepo struct {
	ghRepo      *github.Repository
	cloned      bool
	shallowRepo bool
	local       string
	name        string
	parent      string
	htmlURL     string
	cloneURL    string
	reference   string
}

func NewGithubRepo(ghRepo *github.Repository) *GithubRepo {
	return &GithubRepo{
		ghRepo: ghRepo,
	}
}

func (r *GithubRepo) Name() string {
	if r.name == "" {
		r.name = r.ghRepo.GetFullName()
	}
	return r.name
}

func (r *GithubRepo) Parent() string {
	if r.parent == "" {
		r.parent = r.ghRepo.Parent.GetFullName()
	}
	return r.parent
}

func (r *GithubRepo) ProjectURL() string {
	if r.htmlURL == "" {
		r.htmlURL = r.ghRepo.GetHTMLURL()
	}
	return r.htmlURL
}

func (r *GithubRepo) CloneURL() string {
	if r.cloneURL == "" {
		if config.UseSSH {
			r.cloneURL = r.ghRepo.GetSSHURL()
		} else {
			r.cloneURL = r.ghRepo.GetCloneURL()
		}
	}
	return r.cloneURL
}

func (r *GithubRepo) PathToURL(relativePath string, lineNo int) string {
	return fmt.Sprintf("%s/blob/%s/%s#L%d",
		r.ProjectURL(), r.Reference(), relativePath, lineNo)
}

func (r *GithubRepo) Checkout(target string, shallow bool) (skip bool, err error) {
	if err = gitutils.Checkout(r.CloneURL(), target, shallow); err != nil {
		return
	}

	r.shallowRepo = true
	r.cloned = true
	r.local = target
	r.reference = gitutils.GetReference(r.local)
	return
}

func (r *GithubRepo) Author(relativePath string, lineNo int) (author string, err error) {
	if !r.cloned {
		err = errors.New("can't get author on shallowed repo")
		return
	}

	if r.shallowRepo {
		if err = r.unshallow(); err != nil {
			return
		}
		r.shallowRepo = false
	}

	author = gitutils.LineAuthor(r.local, relativePath, lineNo)
	return
}

func (r *GithubRepo) Owners() (owners []string, err error) {
	if owner := r.ghRepo.GetOwner(); owner != nil {
		if owner.GetType() == "User" && owner.GetLogin() != "" {
			owners = []string{owner.GetLogin()}
		}
	}
	return
}

func (r *GithubRepo) Reference() string {
	if r.reference == "" {
		r.reference = gitutils.GetReference(r.local)
	}
	return r.reference
}

func (r *GithubRepo) unshallow() error {
	return gitutils.Unshallow(r.local)
}

func (r *GithubRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *GithubRepo) GenLocalPath(workDir string) string {
	return path.Join(workDir, r.Name())
}

func (r *GithubRepo) IsPrivate() bool {
	return r.ghRepo.GetPrivate()
}

func (r *GithubRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.name,
		Parent:     r.parent,
		CloneURL:   r.cloneURL,
		ProjectURL: r.htmlURL,
		Owners:     owners,
		Private:    r.IsPrivate(),
	}
}
