package gitlabrepo

import (
	"errors"
	"fmt"
	"path"
	"strings"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/gitlab"
	"a.yandex-team.ru/security/hector/internal/gitutils"
	"a.yandex-team.ru/security/hector/internal/remote"
)

type GitlabRepo struct {
	proj        gitlab.Project
	cloned      bool
	shallowRepo bool
	local       string
	reference   string
}

func NewRepo(project gitlab.Project) *GitlabRepo {
	return &GitlabRepo{
		proj: project,
	}
}

func (r *GitlabRepo) Name() string {
	return r.proj.Name
}

func (r *GitlabRepo) Parent() string {
	return ""
}

func (r *GitlabRepo) ProjectURL() string {
	return r.proj.UIRepo
}

func (r *GitlabRepo) CloneURL() string {
	if config.UseSSH {
		return r.proj.SSHRepo
	}
	return r.proj.HTTPRepo
}

func (r *GitlabRepo) PathToURL(relativePath string, lineNo int) string {
	return fmt.Sprintf("%s/-/blob/%s/%sL#%d",
		r.ProjectURL(), r.Reference(), strings.Trim(relativePath, "/"), lineNo)
}

func (r *GitlabRepo) Checkout(target string, shallow bool) (skip bool, err error) {
	if err = gitutils.Checkout(r.CloneURL(), target, shallow); err != nil {
		return
	}

	r.shallowRepo = true
	r.cloned = true
	r.local = target
	r.reference = gitutils.GetReference(r.local)
	return
}

func (r *GitlabRepo) Author(relativePath string, lineNo int) (author string, err error) {
	// Rewrite to Stash API?
	// https://docs.atlassian.com/DAC/rest/stash/3.11.6/stash-rest.html#idp3106640

	if !r.cloned {
		err = errors.New("can't get author on shallowed repo")
		return
	}

	if r.shallowRepo {
		if err = r.unshallow(); err != nil {
			return
		}
		r.shallowRepo = false
	}

	author = gitutils.LineAuthor(r.local, relativePath, lineNo)
	return
}

func (r *GitlabRepo) Owners() (owners []string, err error) {
	//TODO(Buglloc): implement
	return
}

func (r *GitlabRepo) Reference() string {
	if r.reference == "" {
		r.reference = gitutils.GetReference(r.local)
	}
	return r.reference
}

func (r *GitlabRepo) unshallow() error {
	return gitutils.Unshallow(r.local)
}

func (r *GitlabRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *GitlabRepo) GenLocalPath(workDir string) string {
	return path.Join(workDir, r.Name())
}

func (r *GitlabRepo) IsPrivate() bool {
	return r.proj.Visibility == gitlab.VisibilityPrivate
}

func (r *GitlabRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.Name(),
		Parent:     r.Parent(),
		CloneURL:   r.CloneURL(),
		ProjectURL: r.ProjectURL(),
		Owners:     owners,
		Private:    r.IsPrivate(),
	}
}
