package pasterepo

import (
	"context"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path"
	"regexp"
	"strconv"
	"strings"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/core"
	"a.yandex-team.ru/security/hector/internal/remote"
	"a.yandex-team.ru/security/libs/go/retry"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

var authorRe = regexp.MustCompile(`<span class="yauser"><a href="/([^"]+)"><b>`)

type PasteRepo struct {
	id     int
	author string
}

func NewPasteRepo(id int) *PasteRepo {
	return &PasteRepo{
		id: id,
	}
}

func (r *PasteRepo) Name() string {
	return strconv.Itoa(r.id)
}

func (r *PasteRepo) Parent() string {
	return ""
}

func (r *PasteRepo) ProjectURL() string {
	return fmt.Sprintf("https://paste.yandex-team.ru/%d", r.id)
}

func (r *PasteRepo) CloneURL() string {
	return fmt.Sprintf("https://paste.yandex-team.ru/%d/text", r.id)
}

func (r *PasteRepo) PathToURL(relativePath string, lineNo int) string {
	return fmt.Sprintf("https://paste.yandex-team.ru/%d", r.id)
}

func (r *PasteRepo) Checkout(target string, shallow bool) (skip bool, err error) {
	notFound := false
	work := func(ctx context.Context) error {
		resp, err := core.Client.Get(r.CloneURL())
		if err != nil {
			return fmt.Errorf("failed to get paste: %s", err.Error())
		}
		defer yahttp.GracefulClose(resp.Body)

		if resp.StatusCode == 404 {
			notFound = true
			// Skip not found pastes
			return nil
		}

		if resp.StatusCode != 200 {
			return fmt.Errorf("failed to get paste, status code: %s", resp.Status)
		}

		_ = os.MkdirAll(path.Dir(target), 0755)
		out, err := os.Create(target)
		if err != nil {
			return fmt.Errorf("failed to create resulting paste file: %s", err.Error())
		}
		defer func() { _ = out.Close() }()

		_, err = io.Copy(out, resp.Body)
		if err != nil {
			return fmt.Errorf("failed to save paste: %s", err.Error())
		}
		return nil
	}

	retrier := retry.New(retry.WithAttempts(config.RetryCount))
	if err = retrier.Try(context.Background(), work); err != nil {
		return
	}

	if notFound {
		err = errors.New("paste not found")
		skip = true
		return
	}

	return
}

func (r *PasteRepo) Author(relativePath string, lineNo int) (author string, err error) {
	if r.author != "" {
		author = r.author
		return
	}

	resp, err := core.Client.Get(r.ProjectURL())
	if err != nil {
		err = fmt.Errorf("failed to get paste: %s", err.Error())
		return
	}
	defer yahttp.GracefulClose(resp.Body)

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		err = fmt.Errorf("failed to read paste: %s", err.Error())
		return
	}

	submatches := authorRe.FindSubmatch(body)
	if len(submatches) < 2 {
		err = fmt.Errorf("failed to get author (regexp)")
		return
	}

	author = string(submatches[1])
	r.author = author
	return
}

func (r *PasteRepo) Owners() (owners []string, err error) {
	author, err := r.Author("", 0)
	if err == nil {
		owners = []string{author}
	}
	return
}

func (r *PasteRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *PasteRepo) GenLocalPath(workDir string) string {
	name := r.Name()
	padded, _ := padChar(name, 7, '0')
	result := workDir
	for i := 0; i < 3; i++ {
		result = path.Join(result, string(padded[i]))
	}
	return path.Join(result, name)
}

func (r *PasteRepo) Reference() string {
	return ""
}

func (r *PasteRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.Name(),
		Parent:     "",
		CloneURL:   r.CloneURL(),
		ProjectURL: r.ProjectURL(),
		Owners:     owners,
		Private:    false,
	}
}

func padChar(s string, n int, r rune) (string, error) {
	if n < 0 {
		return "", fmt.Errorf("invalid length %d", n)
	}
	if len(s) > n {
		return s, nil
	}
	return strings.Repeat(string(r), n-len(s)) + s, nil
}
