package pypirepo

import (
	"errors"
	"fmt"
	"io"
	"os"
	"os/exec"
	"path"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/hector/internal/procutils"
	"a.yandex-team.ru/security/hector/internal/remote"
	"a.yandex-team.ru/security/libs/go/pypi"
	"a.yandex-team.ru/security/libs/go/yahttp"
)

type PypiRepo struct {
	pkg      pypi.Package
	release  pypi.Release
	fullName string
	pkgPath  string
}

var httpClient = yahttp.NewClient(yahttp.Config{
	RedirectPolicy: yahttp.RedirectFollowSameOrigin,
	DialTimeout:    2 * time.Second,
	Timeout:        120 * time.Second,
})

func NewPypiRepo(pkg pypi.Package, release pypi.Release) *PypiRepo {
	return &PypiRepo{
		pkg:     pkg,
		release: release,
	}
}

func (r *PypiRepo) Name() string {
	if r.fullName != "" {
		return r.fullName
	}

	r.fullName = fmt.Sprintf("%s/%s", r.pkg.Name(), r.release.Version)
	return r.fullName
}

func (r *PypiRepo) Parent() string {
	return ""
}

func (r *PypiRepo) ProjectURL() string {
	return fmt.Sprintf("https://pypi.yandex-team.ru/dashboard/repositories/default/packages/%s/", r.pkg.Name())
}

func (r *PypiRepo) CloneURL() string {
	return r.release.DownloadURL
}

func (r *PypiRepo) PathToURL(relativePath string, lineNo int) string {
	return fmt.Sprintf("%s#%s:%d", r.release.DownloadURL, relativePath, lineNo)
}

func (r *PypiRepo) Checkout(baseDir string, shallow bool) (skip bool, err error) {
	skip, err = r.download(baseDir)
	if skip || err != nil {
		return
	}
	if err = r.unpack(baseDir); err != nil {
		return
	}
	return
}

func (r *PypiRepo) Author(relativePath string, lineNo int) (author string, err error) {
	return "", nil
}

func (r *PypiRepo) Owners() (owners []string, err error) {
	return []string{""}, nil
}

func (r *PypiRepo) GenResultPath(resultDir string) string {
	baseName := strings.Replace(r.Name(), "/", ":", -1)
	return path.Join(resultDir, baseName+".json")
}

func (r *PypiRepo) GenLocalPath(workDir string) string {
	return path.Join(workDir, r.Name())
}

func (r *PypiRepo) PkgPath(workDir string) string {
	if r.pkgPath != "" {
		return r.pkgPath
	}
	r.pkgPath = path.Join(workDir, path.Base(r.CloneURL()))
	return r.pkgPath
}

func (r *PypiRepo) Reference() string {
	return r.Name()
}

func (r *PypiRepo) Export() remote.ExportedRepo {
	owners, _ := r.Owners()

	return remote.ExportedRepo{
		Name:       r.Name(),
		Parent:     "",
		CloneURL:   r.CloneURL(),
		ProjectURL: r.ProjectURL(),
		Owners:     owners,
		Private:    false,
	}
}

func (r *PypiRepo) download(baseDir string) (skip bool, err error) {
	notFound := false

	cloneURL := r.CloneURL()
	target := r.PkgPath(baseDir)
	work := func() error {
		resp, err := httpClient.Get(cloneURL)
		if err != nil {
			return fmt.Errorf("failed to get pkg: %s", err.Error())
		}
		defer yahttp.GracefulClose(resp.Body)

		if resp.StatusCode == 404 {
			notFound = true
			// Skip not found pastes
			return nil
		}

		if resp.StatusCode != 200 {
			return fmt.Errorf("failed to get pkg, status code: %s", resp.Status)
		}

		err = os.MkdirAll(baseDir, 0755)
		if err != nil {
			return fmt.Errorf("failed to create pkg folder: %s", err.Error())
		}

		out, err := os.Create(target)
		if err != nil {
			return fmt.Errorf("failed to create resulting pkg file: %s", err.Error())
		}
		defer func() { _ = out.Close() }()

		_, err = io.Copy(out, resp.Body)
		if err != nil {
			return fmt.Errorf("failed to pkg paste: %s", err.Error())
		}
		return nil
	}

	// TODO(buglloc): enable retries back?
	if err = work(); err != nil {
		return
	}

	if notFound {
		err = errors.New("pkg not found")
		skip = true
		return
	}
	return
}

func (r *PypiRepo) unpack(baseDir string) error {
	var cmd *exec.Cmd
	switch r.release.Ext {
	case ".tar.bz2", ".tbz", ".tbz2":
		cmd = exec.Command("tar", "xjf", r.PkgPath(baseDir))
	case ".tar.gz", ".tgz":
		cmd = exec.Command("tar", "xzf", r.PkgPath(baseDir))
	case ".zip", ".whl", ".egg":
		cmd = exec.Command("unzip", r.PkgPath(baseDir))
	case ".tar":
		cmd = exec.Command("tar", "xf", r.PkgPath(baseDir))
	default:
		return xerrors.Errorf("unknown arch type: %s", r.release.Ext)
	}
	cmd.Dir = baseDir
	res, err := procutils.RunCmd(cmd)
	if err != nil {
		return xerrors.Errorf("failed to unpack: %w", err)
	}

	if res.ExitCode != 0 {
		return xerrors.Errorf("failed to unpack (exit code = %d): %s", res.ExitCode, res.Stderr)
	}

	return nil
}
