package stash

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"

	"a.yandex-team.ru/security/hector/internal/config"
	"a.yandex-team.ru/security/hector/internal/core"
	"a.yandex-team.ru/security/libs/go/retry"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	Stash interface {
		GetRepositories() (map[int]Repository, error)
		GetRepository(projectKey, repositorySlug string) (Repository, error)
		WalkRepositories(RepoWalkFn) error
		WalkProjectRepositories(string, RepoWalkFn) error
		HaveCommitsSince(projectKey, repositorySlug string, since string) (bool, error)
	}

	RepoWalkFn func(repo Repository) (next bool)

	Client struct {
		userName string
		password string
		baseURL  *url.URL
		retrier  retry.Retrier
		Stash
	}

	Page struct {
		IsLastPage    bool `json:"isLastPage"`
		Size          int  `json:"size"`
		Start         int  `json:"start"`
		NextPageStart int  `json:"nextPageStart"`
	}

	Repositories struct {
		IsLastPage    bool         `json:"isLastPage"`
		Size          int          `json:"size"`
		Start         int          `json:"start"`
		NextPageStart int          `json:"nextPageStart"`
		Repository    []Repository `json:"values"`
	}

	Repository struct {
		ID      int         `json:"id"`
		Name    string      `json:"name"`
		Slug    string      `json:"slug"`
		Origin  *Repository `json:"origin"`
		Project Project     `json:"project"`
		ScmID   string      `json:"scmId"`
		Links   Links       `json:"links"`
		Public  bool        `json:"public"`
	}

	Project struct {
		Key string `json:"key"`
	}

	Links struct {
		Clones []Clone `json:"clone"`
		Self   []Self  `json:"self"`
	}

	Clone struct {
		HREF string `json:"href"`
		Name string `json:"name"`
	}

	Self struct {
		HREF string `json:"href"`
	}

	Ref struct {
		DisplayID string `json:"displayId"`
	}

	Commit struct {
		ID        string `json:"id"`
		DisplayID string `json:"displayId"`
		Author    struct {
			Name         string `json:"name"`
			EmailAddress string `json:"emailAddress"`
		} `json:"author"`
		AuthorTimestamp int64 `json:"authorTimestamp"` // in milliseconds since the epoch
	}

	Commits struct {
		Commits []Commit `json:"values"`
	}

	errorResponse struct {
		StatusCode int
		Reason     string
		error
	}

	stashError struct {
		Errors []struct {
			Context       string `json:"context"`
			Message       string `json:"message"`
			ExceptionName string `json:"exceptionName"`
		} `json:"errors"`
	}
)

const (
	stashPageLimit = 25
)

func (e errorResponse) Error() string {
	return fmt.Sprintf("%s (%d)", e.Reason, e.StatusCode)
}

func NewClient(userName, password string, baseURL *url.URL) Stash {
	return Client{
		userName: userName,
		password: password,
		baseURL:  baseURL,
		retrier:  retry.New(retry.WithAttempts(config.RetryCount)),
	}
}

// GetRepositories returns a map of repositories indexed by repository URL.
func (client Client) GetRepositories() (map[int]Repository, error) {
	start := 0
	repositories := make(map[int]Repository)
	morePages := true
	for morePages {
		uri := fmt.Sprintf("%s/rest/api/1.0/repos?start=%d&limit=%d",
			client.baseURL.String(),
			start,
			stashPageLimit)

		data, err := client.makeRequest(uri)
		if err != nil {
			return nil, err
		}

		var r Repositories
		err = json.Unmarshal(data, &r)
		if err != nil {
			return nil, err
		}
		for _, repo := range r.Repository {
			repositories[repo.ID] = repo
		}
		morePages = !r.IsLastPage
		start = r.NextPageStart
	}
	return repositories, nil
}

// GetRepository returns a repository representation for the given Stash Project key and repository slug.
func (client Client) GetRepository(projectKey, repositorySlug string) (Repository, error) {
	uri := fmt.Sprintf("%s/rest/api/1.0/projects/%s/repos/%s",
		client.baseURL.String(),
		projectKey,
		repositorySlug)

	var r Repository
	data, err := client.makeRequest(uri)
	if err != nil {
		return r, err
	}
	err = json.Unmarshal(data, &r)
	return r, err
}

// WalkRepositories walk though stash repositories.
func (client Client) WalkRepositories(walkFn RepoWalkFn) error {
	start := 0
	morePages := true
	for morePages {
		uri := fmt.Sprintf("%s/rest/api/1.0/repos?start=%d&limit=%d",
			client.baseURL.String(),
			start,
			stashPageLimit)

		data, err := client.makeRequest(uri)
		if err != nil {
			return err
		}

		var r Repositories
		err = json.Unmarshal(data, &r)
		if err != nil {
			return err
		}
		for _, repo := range r.Repository {
			if !walkFn(repo) {
				return nil
			}
		}
		morePages = !r.IsLastPage
		start = r.NextPageStart
	}
	return nil
}

// WalkRepositories walk though stash project repositories.
func (client Client) WalkProjectRepositories(project string, walkFn RepoWalkFn) error {
	start := 0
	morePages := true
	for morePages {
		uri := fmt.Sprintf("%s/rest/api/1.0/projects/%s/repos?start=%d&limit=%d",
			client.baseURL.String(),
			project,
			start,
			stashPageLimit)

		data, err := client.makeRequest(uri)
		if err != nil {
			return err
		}

		var r Repositories
		err = json.Unmarshal(data, &r)
		if err != nil {
			return err
		}
		for _, repo := range r.Repository {
			if !walkFn(repo) {
				return nil
			}
		}
		morePages = !r.IsLastPage
		start = r.NextPageStart
	}
	return nil
}

// HaveCommitsSince returns if repo have changes since date.
func (client Client) HaveCommitsSince(projectKey, repositorySlug string, since string) (haveCommits bool, err error) {
	uri := fmt.Sprintf("%s/rest/api/1.0/projects/%s/repos/%s/commits?limit=1&since=%s",
		client.baseURL.String(),
		projectKey,
		repositorySlug,
		since)

	data, err := client.makeRequest(uri)
	if err != nil {
		return
	}

	var commits Commits
	err = json.Unmarshal(data, &commits)
	if err != nil {
		return
	}

	haveCommits = len(commits.Commits) > 0
	return
}

// SSHURL extracts the SSH-based URL from the repository metadata.
func (repo Repository) SSHURL() string {
	for _, clone := range repo.Links.Clones {
		if clone.Name == "ssh" {
			return clone.HREF
		}
	}
	return ""
}

// HTTPURL extracts the HTTP-based URL from the repository metadata.
func (repo Repository) HTTPURL() string {
	for _, clone := range repo.Links.Clones {
		if clone.Name == "http" {
			return clone.HREF
		}
	}
	return ""
}

// HTMLURL extracts the HTML URL from the repository metadata.
func (repo Repository) HTMLURL() string {
	for _, self := range repo.Links.Self {
		if self.HREF != "" {
			return self.HREF
		}
	}
	return ""
}

// Name construct repo name from the repository metadata.
func (repo Repository) FullName() string {
	return fmt.Sprintf("%s/%s", repo.Project.Key, repo.Name)
}

// IsFork returns true if repo is fork of another.
func (repo Repository) IsFork() bool {
	return repo.Origin != nil
}

func (client Client) consumeResponse(req *http.Request) (int, []byte, error) {
	response, err := core.Client.Do(req)
	if err != nil {
		return 0, nil, err
	}

	data, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return response.StatusCode, nil, err
	}

	defer func() {
		_, _ = io.Copy(ioutil.Discard, response.Body)
		if err := response.Body.Close(); err != nil {
			simplelog.Warn("stash: error closing response body", "err", err)
		}
	}()

	if response.StatusCode >= 400 {
		// https://jira.xoom.com/browse/AS-42
		contentType := response.Header.Get("Content-type")
		if !strings.HasPrefix(contentType, "application/json") {
			return response.StatusCode, data, nil
		}

		var errResponse stashError
		if err := json.Unmarshal(data, &errResponse); err == nil {
			var messages []string
			for _, e := range errResponse.Errors {
				messages = append(messages, e.Message)
			}
			return response.StatusCode, data, errors.New(strings.Join(messages, " "))
		}
		return response.StatusCode, nil, err
	}
	return response.StatusCode, data, nil
}

func (client Client) makeRequest(uri string) ([]byte, error) {
	var data []byte
	work := func(ctx context.Context) error {
		req, err := http.NewRequest("GET", uri, nil)
		if err != nil {
			return err
		}
		req.Header.Set("Accept", "application/json")
		// use credentials if we have them.  If not, the repository must be public.
		if client.userName != "" && client.password != "" {
			req.SetBasicAuth(client.userName, client.password)
		}

		var responseCode int
		responseCode, data, err = client.consumeResponse(req)
		if err != nil {
			return err
		}
		if responseCode != http.StatusOK {
			var reason = "unhandled reason"
			switch {
			case responseCode == http.StatusBadRequest:
				reason = "Bad request."
			}
			return errorResponse{StatusCode: responseCode, Reason: reason}
		}
		return nil
	}

	if err := client.retrier.Try(context.Background(), work); err != nil {
		return nil, err
	}

	return data, nil
}
