package state

import (
	"encoding/json"
	"io/ioutil"
	"sync"
	"time"
)

type (
	Repo struct {
		Reference string `json:"ref,omitempty"`
	}

	RepoState struct {
		mx     sync.RWMutex
		Info   map[string]Repo `json:"info"`
		Latest int64           `json:"latest,omitempty"`
	}

	State struct {
		InitAt    int64                 `json:"init_at"`
		ReposInfo map[string]*RepoState `json:"repos"`
	}
)

func NewState() *State {
	return &State{
		InitAt:    time.Now().Unix(),
		ReposInfo: make(map[string]*RepoState),
	}
}

func NewRepoState() *RepoState {
	return &RepoState{
		Info: make(map[string]Repo),
	}
}

func Copy(state *State) *State {
	return &State{
		InitAt:    state.InitAt,
		ReposInfo: make(map[string]*RepoState),
	}
}

func Load(filePath string) (state *State, err error) {
	data, err := ioutil.ReadFile(filePath)
	if err != nil {
		return
	}

	err = json.Unmarshal(data, &state)
	return
}

func Save(filePath string, state *State) (err error) {
	data, err := json.Marshal(state)
	if err != nil {
		return
	}

	err = ioutil.WriteFile(filePath, data, 0644)
	return
}

func (r *State) GetRepoState(name string) *RepoState {
	val, ok := r.ReposInfo[name]
	if !ok {
		repo := NewRepoState()
		r.ReposInfo[name] = repo
		return repo
	}

	return val
}

func (r *RepoState) Load(key string) (Repo, bool) {
	r.mx.Lock()
	defer r.mx.Unlock()
	val, ok := r.Info[key]
	return val, ok
}

func (r *RepoState) StoreRaw(key string, value Repo) {
	r.mx.Lock()
	defer r.mx.Unlock()
	r.Info[key] = value
}

func (r *RepoState) Store(name, ref string) {
	r.StoreRaw(name, Repo{
		Reference: ref,
	})
}
