#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "argpatch.h"
#include "libhfcommon/log.h"

/**
 * Skip spaces in string of size max_len starting from the given position
 * Overwrites position
 */
void skip_spaces(ulong *pos, const char *str, size_t max_len) {
    while (*pos < max_len && isspace(str[*pos])) {
        (*pos)++;
    }
}

/**
 * Skips spaces and gets CL argument from str, starting from pos index
 */
char *get_arg(ulong *pos, const char *str, size_t max_len) {
    skip_spaces(pos, str, max_len);
    if (*pos == max_len) {
        return NULL;
    }

    ulong start = *pos;
    while (*pos < max_len && !isspace(str[*pos])) {
        (*pos)++;
    } // pos points to space or ending

    size_t arg_len = *pos - start;
    char *entry = (char *) malloc(sizeof(char) * (arg_len + 1)); // +1 - '\0' terminator
    if (entry == NULL) {
        return NULL;
    }
    memcpy(entry, &str[start], arg_len);
    entry[arg_len] = '\0';
    return entry;
}

/**
 * Count arguments in a str of max_len
 * @return number of CL arguments
 */
ulong count_args(const char *str, size_t max_len) {
    ulong pos = 0;
    for (ulong i = 0;; i++) {
        char *arg = get_arg(&pos, str, max_len);
        if (arg == NULL) {
            return i;
        }
        free(arg);
    }
}

/**
 * Free method for struct Args
 */
void free_argstruct(ArgStruct *args) {
    if (args == NULL || args->argv == NULL) {
        return;
    }
    for (int i = 0; i < args->argc; i++) {
        if (args->argv[i] != NULL) {
            free(args->argv[i]);
        }
    }
    free(args->argv);
}

ArgStruct default_argstruct(size_t size) {
    ArgStruct argstruct = {size, calloc(size, sizeof(char *))};
    if (argstruct.argv == NULL) {
        argstruct.argc = 0;
        return argstruct;
    }
    if (prog_name[0] == '\0') {
        LOG_F("Unable to determine program name!");
        argstruct.argc = 0;
        free(argstruct.argv);
        return argstruct;
    }
    argstruct.argv[0] = strdup(prog_name);
    return argstruct;
}

/**
 * @return ArgStruct containing CL arguments extracted from environment variable
 */
ArgStruct get_args_from_env() {
    const char *args = getenv(PROG_ARGS_ENV_VAR);
    if (args == NULL) {
        return default_argstruct(1);
    }

    size_t arg_size = strnlen(args, PROG_MAX_ARG_SIZE);
    if (arg_size == PROG_MAX_ARG_SIZE) {
        LOG_W("Program arguments may be truncated!");
    }

    ulong argc = count_args(args, arg_size) + 1; //first is prog name
    ArgStruct argstruct = default_argstruct(argc);
    if (argstruct.argv == NULL) {
        return argstruct;
    }

    ulong i = 0;
    ulong arg_index = 1; // 0 is prog_name
    for ever {
        char *arg = get_arg(&i, args, arg_size);
        if (arg == NULL) {
            break;
        }
        if (arg_index >= argc) {
            LOG_F("Memory corrupted! arg_index > argc");
            free(arg);
            return argstruct;
        }
        argstruct.argv[arg_index] = arg;
        arg_index++;
    }
    return argstruct;
}

/**
 * set
 * @param argv0
 */
void set_argv_0(const char *argv0) {
    size_t cp_len = strnlen(argv0, PROG_NAME_LIMIT - 1);
    memcpy(prog_name, argv0, cp_len);
    prog_name[cp_len] = '\0';
}
