package codeql

import (
	"database/sql"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/codeql-api/internal/infra"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_codeQLDatabaseRepository "a.yandex-team.ru/security/impulse/api/repositories/codeqldb"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_codeQLUsecase "a.yandex-team.ru/security/impulse/api/usecases/codeqldb"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	projectUsecase        _projectUsecase.Usecase
	codeQLDatabaseUsecase _codeQLUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	scanInstanceUsecase := _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, scanInstanceUsecase)
	codeQLDatabaseRepo := _codeQLDatabaseRepository.NewCodeQLDatabaseRepository(c.DB)
	c.codeQLDatabaseUsecase = _codeQLUsecase.NewCodeQLDatabaseUsecase(codeQLDatabaseRepo)

	g.GET("/organizations/:organizationId/projects/:projectId/languages/:language/tags/:tag",
		c.getCodeQLDBInfo)
	g.PUT("/organizations/:organizationId/projects/:projectId/languages/:language/tags/:tag",
		c.saveCodeQLDBInfo)
	g.DELETE("/organizations/:organizationId/projects/:projectId/languages/:language/tags/:tag",
		c.deleteCodeQLDBInfo)
	g.GET("/organizations/:organizationId/projects/:projectId/languages/:language/tags",
		c.getCodeQLDBtags)
	g.POST("/search_by_mds_url", c.searchByMdsURL)

	return nil
}

func (c *Controller) getCodeQLDBInfo(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "language", e.Param("language"), "tag",
		e.Param("tag"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByOrganizationIDAndProjectID(e.Request().Context(), organizationID,
		projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	language := e.Param("language")
	tag := e.Param("tag")

	codeQLDatabaseArchive, err := c.codeQLDatabaseUsecase.Get(e.Request().Context(), *project,
		language, tag)
	if err == sql.ErrNoRows {
		return utils.APINotFound(e)
	} else if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, codeQLDatabaseArchive)
}

func (c *Controller) saveCodeQLDBInfo(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "language", e.Param("language"), "tag",
		e.Param("tag"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByOrganizationIDAndProjectID(e.Request().Context(), organizationID,
		projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	language := e.Param("language")
	tag := e.Param("tag")
	codeQLDatabaseSaveRequestDTO := new(models.CodeQLDatabaseSaveRequestDTO)
	if err := e.Bind(codeQLDatabaseSaveRequestDTO); err != nil {
		return utils.APIError(e, err)
	}
	simplelog.Info(e.Request().Method+" "+e.Path(), "mds_url", codeQLDatabaseSaveRequestDTO.MdsURL,
		"revision", codeQLDatabaseSaveRequestDTO.Revision)

	codeQLDatabaseArchive, err := c.codeQLDatabaseUsecase.Save(e.Request().Context(), *project, language, tag,
		codeQLDatabaseSaveRequestDTO.MdsURL, codeQLDatabaseSaveRequestDTO.Revision)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, codeQLDatabaseArchive)
}

func (c *Controller) deleteCodeQLDBInfo(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "language", e.Param("language"), "tag",
		e.Param("tag"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByOrganizationIDAndProjectID(e.Request().Context(), organizationID,
		projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	language := e.Param("language")
	tag := e.Param("tag")

	codeQLDatabaseArchive, err := c.codeQLDatabaseUsecase.Get(e.Request().Context(), *project,
		language, tag)
	if err == sql.ErrNoRows {
		return utils.APINotFound(e)
	} else if err != nil {
		return utils.APIError(e, err)
	}

	err = c.codeQLDatabaseUsecase.Delete(e.Request().Context(), *codeQLDatabaseArchive)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) getCodeQLDBtags(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "language", e.Param("language"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByOrganizationIDAndProjectID(e.Request().Context(), organizationID,
		projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	language := e.Param("language")

	tags, err := c.codeQLDatabaseUsecase.ListTags(e.Request().Context(), *project,
		language)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, tags)
}

func (c *Controller) searchByMdsURL(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	codeQLDatabaseSearchRequestDTO := new(models.CodeQLDatabaseSearchRequestDTO)
	if err := e.Bind(codeQLDatabaseSearchRequestDTO); err != nil {
		return utils.APIError(e, err)
	}

	codeQLDatabaseArchive, err := c.codeQLDatabaseUsecase.GetByMDS(e.Request().Context(),
		codeQLDatabaseSearchRequestDTO.MdsURL)
	if err == sql.ErrNoRows {
		return utils.APINotFound(e)
	} else if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, codeQLDatabaseArchive)
}
