package main

import (
	"context"
	"os"
	"os/signal"
	"strconv"
	"syscall"
	"time"

	"a.yandex-team.ru/security/impulse/api/codeql-api/internal/config"
	"a.yandex-team.ru/security/impulse/api/codeql-api/internal/infra"
	"a.yandex-team.ru/security/impulse/api/codeql-api/internal/server"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	shutdownDeadline = time.Second * 60
)

func newConfig() config.Config {
	isDebug := false
	useAuth := true

	switch os.Getenv("IMPULSE_ENV") {
	case "dev":
		isDebug = true
		useAuth = false
	case "test":
		isDebug = false
	}

	if isDebug {
		simplelog.SetLevel(simplelog.DebugLevel)
	}
	httpPort, err := strconv.Atoi(os.Getenv("HTTP_PORT"))
	if err != nil {
		httpPort = 80
	}
	cfg := config.Config{
		DBConn:     os.Getenv("DB_DSN"),
		DBPassword: os.Getenv("DB_PASSWORD"),
		DBRetries:  3,
		HTTPPort:   uint32(httpPort),
		UseAuth:    useAuth,
		Debug:      isDebug,
	}
	return cfg
}

func main() {
	cfg := newConfig()
	srv, err := server.New(infra.New(cfg))
	if err != nil {
		panic(err)
	}

	quit := make(chan os.Signal)

	go func() {
		if err := srv.ListenAndServe(cfg.HTTPPort); err != nil {
			simplelog.Error("server stopped", "err", err)
			signal.Stop(quit)
			quit <- syscall.SIGTERM
		}
	}()

	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
	defer cancel()

	if err := srv.Shutdown(ctx); err != nil {
		simplelog.Error("failed to shutdown server", "err", err)
	}
}
