package abc

import (
	"fmt"
	"net/url"

	"github.com/go-resty/resty/v2"
	"golang.org/x/xerrors"
)

type (
	API struct {
		Client *resty.Client
	}

	Options struct {
		Endpoint   string
		OAuthToken string
	}

	Members struct {
		Next     string `json:"next"`
		Previous string `json:"previous"`
		Results  []struct {
			Person struct {
				Login string `json:"login"`
			} `json:"person"`
		} `json:"results"`
	}

	AllowDutys struct {
		Count      int    `json:"count"`
		Next       string `json:"next"`
		Previous   string `json:"previous"`
		TotalPages int    `json:"total_pages"`
		Results    []struct {
			Login string `json:"login"`
		} `json:"results"`
	}

	OnDuty []struct {
		Person struct {
			Login string `json:"login"`
		} `json:"person"`
		StartTime string `json:"start_datetime"`
		EndTine   string `json:"end_datetime"`
	}
)

func New(opts *Options) *API {
	httpc := resty.New().
		SetHeader("Content-Type", "application/json; charset=utf-8").
		SetHeader("Authorization", fmt.Sprintf("OAuth %s", opts.OAuthToken)).
		SetBaseURL(opts.Endpoint)

	return &API{
		Client: httpc,
	}
}

func (api *API) Close() error {
	return nil
}

func (api *API) GetMembers(serviceID int, withDescendants bool, scope string) ([]string, error) {
	queryParams := make(map[string]string)
	if withDescendants {
		queryParams["service__with_descendants"] = fmt.Sprintf("%d", serviceID)
	} else {
		queryParams["service"] = fmt.Sprintf("%d", serviceID)
	}
	if scope != "" {
		queryParams["role__scope"] = scope
	}
	queryParams["page_size"] = "1000"
	queryParams["fields"] = "person.login"

	var logins []string
	for {
		var resp Members
		httpResp, err := api.Client.R().
			SetQueryParams(queryParams).
			SetResult(&resp).
			Get("api/v4/services/members")

		if err != nil {
			return nil, err
		}

		if !httpResp.IsSuccess() {
			return nil, xerrors.Errorf("wrong status code: %s", httpResp.Status())
		}
		for _, p := range resp.Results {
			logins = append(logins, p.Person.Login)
		}
		if resp.Next == "" {
			break
		}
		u, err := url.Parse(resp.Next)
		if err != nil {
			break
		}
		queryParams["cursor"] = u.Query().Get("cursor")
	}

	return logins, nil
}

func (api *API) GetDutys(serviceID int, onDuty bool) ([]string, error) {
	queryParams := make(map[string]string)
	queryParams["service"] = fmt.Sprintf("%d", serviceID)
	var logins []string

	if onDuty {
		var resp OnDuty
		queryParams["fields"] = "person.login"
		httpResp, err := api.Client.R().
			SetQueryParams(queryParams).
			SetResult(&resp).
			Get("api/v4/duty/on_duty/")

		if err != nil {
			return nil, err
		}

		if !httpResp.IsSuccess() {
			return nil, xerrors.Errorf("wrong status code: %s", httpResp.Status())
		}
		for _, p := range resp {
			logins = append(logins, p.Person.Login)
		}
	} else {
		for {
			var resp AllowDutys
			queryParams["fields"] = "login"
			httpResp, err := api.Client.R().
				SetQueryParams(queryParams).
				SetResult(&resp).
				Get("api/v4/duty/allowforduty/")
			if err != nil {
				return nil, err
			}

			if !httpResp.IsSuccess() {
				return nil, xerrors.Errorf("wrong status code: %s", httpResp.Status())
			}

			for _, p := range resp.Results {
				logins = append(logins, p.Login)
			}
			if resp.Next == "" {
				break
			}
			u, err := url.Parse(resp.Next)
			if err != nil {
				break
			}
			queryParams["cursor"] = u.Query().Get("cursor")
		}
	}

	return logins, nil
}
