package callback

import (
	"bytes"
	"context"
	"net/http"
	"net/url"
	"strconv"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
)

type API interface {
	Do(ctx context.Context, callbackURL string, data []byte) error
	SetTVMClient(tvmClient *tvmtool.Client)
}

type (
	apiImpl struct {
		Client    *http.Client
		TvmClient *tvmtool.Client
	}
)

func New() *apiImpl {
	return &apiImpl{
		Client: &http.Client{},
	}
}

func getTvmID(callbackURLString string) (tvm.ClientID, error) {
	callbackURL, err := url.Parse(callbackURLString)
	if err != nil {
		return 0, err
	}
	params, ok := callbackURL.Query()["tvm_id"]
	if !ok || len(params) < 1 {
		return 0, nil
	} else {
		tvmID, err := strconv.Atoi(params[0])
		if err != nil {
			return 0, err
		}
		return tvm.ClientID(tvmID), nil
	}
}

func (api *apiImpl) SetTVMClient(tvmClient *tvmtool.Client) {
	api.TvmClient = tvmClient
}

func (api *apiImpl) Do(ctx context.Context, callbackURL string, data []byte) error {
	tvmID, err := getTvmID(callbackURL)
	if err != nil {
		return err
	}
	req, _ := http.NewRequest("POST", callbackURL, bytes.NewBuffer(data))
	req.Header.Add("Content-Type", "application/json; charset=utf-8")

	if tvmID != 0 && api.TvmClient != nil {
		ticket, err := api.TvmClient.GetServiceTicketForID(ctx, tvmID)
		if err != nil {
			return xerrors.Errorf("failed to receive TVM ticket for %d: %w", tvmID, err)
		}
		req.Header.Add("X-Ya-Service-Ticket", ticket)
	}

	resp, err := api.Client.Do(req)
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()
	if resp.StatusCode != http.StatusOK {
		return xerrors.Errorf("Wrong status code %d", resp.StatusCode)
	}

	return nil
}
