package db

import (
	"time"

	"github.com/jackc/pgx/v4"
	pgxstd "github.com/jackc/pgx/v4/stdlib"
	"github.com/jmoiron/sqlx"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/security/libs/go/retry"
	"a.yandex-team.ru/security/libs/go/retry/backoff"
)

type (
	DB struct {
		PG    *sqlx.DB
		Trier retry.Retrier
	}

	Options struct {
		DSN             string
		Password        string
		MaxOpenConns    int
		MaxIdleConns    int
		ConnMaxLifetime time.Duration
		QueryRetries    int
	}
)

func New(opts *Options) (*DB, error) {
	config, err := pgx.ParseConfig(opts.DSN)
	if err != nil {
		err = xerrors.Errorf("failed to create db: %w", err)
		return nil, err
	}

	caCertPool, err := certifi.NewCertPoolInternal()

	if err != nil {
		err = xerrors.Errorf("failed to internal CA cert pool: %w", err)
		return nil, err
	}

	config.TLSConfig.RootCAs = caCertPool
	config.PreferSimpleProtocol = true
	if opts.Password != "" {
		config.Password = opts.Password
	}

	if opts.MaxOpenConns <= 0 {
		opts.MaxOpenConns = 50
	}

	driverDB := pgxstd.OpenDB(*config)
	pg := sqlx.NewDb(driverDB, "pgx")
	pg.SetMaxOpenConns(opts.MaxOpenConns)
	pg.SetMaxIdleConns(opts.MaxIdleConns)
	pg.SetConnMaxLifetime(opts.ConnMaxLifetime)

	err = pg.Ping()
	if err != nil {
		err = xerrors.Errorf("failed to ping db: %w", err)
		return nil, err
	}

	db := DB{
		PG: pg,
		Trier: retry.New(
			retry.WithAttempts(opts.QueryRetries),
			retry.WithBackOff(backoff.NewExponential(100*time.Millisecond, 2*time.Second)),
			retry.WithCondition(retryStopFunc),
		),
	}

	if err := db.Ping(); err != nil {
		return nil, xerrors.Errorf("ping failed: %w", err)
	}

	return &db, nil
}

func (db *DB) Close() error {
	err := db.PG.Close()
	if err != nil {
		return xerrors.Errorf("failed to close db connections: %w", err)
	}

	return nil
}
