package db

import (
	"context"
	"database/sql"
)

var (
	ErrNotFound = sql.ErrNoRows
)

func retryStopFunc(err error, attempt int) bool {
	return IsRetriableError(err)
}

func IsRetriableError(err error) bool {
	return err != ErrNotFound && err != context.Canceled
}

func (db *DB) Ping() error {
	sql := `SELECT 1`
	_, err := db.PG.Exec(sql)
	return err
}

func (db *DB) InitSchema() error {
	sql := `
CREATE TABLE IF NOT EXISTS organization (
    id SERIAL NOT NULL PRIMARY KEY,
    name VARCHAR(128) NOT NULL,
    slug VARCHAR(128) UNIQUE NOT NULL,
    tracker_queue VARCHAR(128) NOT NULL DEFAULT '',
    abc_service_id INTEGER NOT NULL DEFAULT 0,
    deleted BOOL DEFAULT 'f'
);

CREATE TABLE IF NOT EXISTS project (
	id SERIAL NOT NULL PRIMARY KEY,
	name VARCHAR(128) NOT NULL,
	slug VARCHAR(128) UNIQUE NOT NULL,
	organization_id INTEGER NOT NULL,
    tracker_queue VARCHAR(128) NOT NULL DEFAULT '',
    abc_service_id INTEGER NOT NULL DEFAULT 0,
    tags VARCHAR(64)[] DEFAULT '{}'::VARCHAR(64)[],
    notification_settings JSON,
    build_codeql_index BOOL DEFAULT 'f',
    deleted BOOL DEFAULT 'f',
	FOREIGN KEY (organization_id) REFERENCES organization (id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS scanType (
	id SERIAL NOT NULL PRIMARY KEY,
	type_name VARCHAR(32) UNIQUE NOT NULL,
	title VARCHAR(32) NOT NULL
);

CREATE TABLE IF NOT EXISTS scanParameter (
	scan_type_id INTEGER NOT NULL,
	name VARCHAR(64) NOT NULL,
	label VARCHAR(256) NOT NULL,
	component VARCHAR(256) NOT NULL,
	properties JSON,
	non_template_parameter BOOL DEFAULT 'f',
	FOREIGN KEY (scan_type_id) REFERENCES scanType (id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS task (
	task_id VARCHAR(36) NOT NULL PRIMARY KEY,
	organization_id INTEGER NOT NULL,
	project_id INTEGER NOT NULL,
	workflow_id VARCHAR(36),
	workflow_instance_id VARCHAR(36),
	parameters JSON,
	analysers JSON,
	sandbox_task_id INTEGER,
	start_time INTEGER,
	end_time INTEGER,
	status INTEGER,
	cron_id INTEGER NOT NULL,
	callback_url VARCHAR(256),
	non_template_scan BOOL DEFAULT 'f',
	FOREIGN KEY (organization_id) REFERENCES organization (id) ON DELETE CASCADE,
	FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS task_organization_id ON task (organization_id);
CREATE INDEX IF NOT EXISTS task_project_id ON task (project_id);
CREATE INDEX IF NOT EXISTS task_workflow_id ON task (workflow_id);
CREATE INDEX IF NOT EXISTS task_status ON task (status);

CREATE TABLE IF NOT EXISTS cron (
	id SERIAL NOT NULL PRIMARY KEY,
	spec VARCHAR(64) NOT NULL,
	next_time INTEGER,
	is_running BOOL DEFAULT 'f',
	description TEXT NOT NULL,
	organization_id INTEGER NOT NULL,
	project_id INTEGER NOT NULL,
	workflow_id VARCHAR(36) NOT NULL,
	parameters JSON,
	analysers JSON,
	callback_url VARCHAR(256),
	non_template_scan BOOL DEFAULT 'f',
	FOREIGN KEY (organization_id) REFERENCES organization (id) ON DELETE CASCADE,
	FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS cron_organization_id ON cron (organization_id);
CREATE INDEX IF NOT EXISTS cron_project_id ON cron (project_id);
CREATE INDEX IF NOT EXISTS cron_workflow_id ON cron (workflow_id);

CREATE TABLE IF NOT EXISTS workflow (
	id VARCHAR(36) NOT NULL PRIMARY KEY,
	name TEXT NOT NULL,
	description TEXT,
	url VARCHAR(256)
);

CREATE TABLE IF NOT EXISTS workflow2scanType (
	workflow_id VARCHAR(36) NOT NULL,
	scan_type_id INTEGER NOT NULL,
	FOREIGN KEY (workflow_id) REFERENCES workflow (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_type_id) REFERENCES scanType (id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS scan (
	id SERIAL NOT NULL PRIMARY KEY,
	project_id INTEGER NOT NULL,
	scan_type_id INTEGER NOT NULL,
	last_update_token uuid NOT NULL,
	UNIQUE (project_id, scan_type_id),
	FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_type_id) REFERENCES scanType (id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS scanInstance (
	id SERIAL NOT NULL PRIMARY KEY,
	scan_id INTEGER NOT NULL,
	task_id VARCHAR(36),
	raw_report_url VARCHAR(256),
	report_url VARCHAR(256),
	commit_hash VARCHAR(40),
	start_time TIMESTAMP,
	end_time TIMESTAMP,
	FOREIGN KEY (scan_id) REFERENCES scan (id) ON DELETE CASCADE ,
	FOREIGN KEY (task_id) REFERENCES task (task_id) ON DELETE CASCADE
);

DO
$$
BEGIN
	IF NOT EXISTS (SELECT * FROM pg_type typ INNER JOIN pg_namespace nsp ON nsp.oid = typ.typnamespace
					WHERE nsp.nspname = current_schema() AND typ.typname = 'severity_type')
	THEN
		CREATE TYPE SEVERITY_TYPE AS ENUM ('blocker', 'critical', 'medium', 'low', 'info');
	END IF;
END;
$$
LANGUAGE plpgsql;

DO
$$
BEGIN
	IF NOT EXISTS (SELECT * FROM pg_type typ INNER JOIN pg_namespace nsp ON nsp.oid = typ.typnamespace
					WHERE nsp.nspname = current_schema() AND typ.typname = 'status_type')
	THEN
		CREATE TYPE STATUS_TYPE AS ENUM ('not_reviewed', 'to_verify', 'confirmed', 'not_exploitable', 'not_an_issue');
	END IF;
END;
$$
LANGUAGE plpgsql;

CREATE TABLE IF NOT EXISTS vulnerabilityCategory (
	id SERIAL NOT NULL PRIMARY KEY,
	scan_type_id INTEGER NOT NULL,
	name VARCHAR(512) NOT NULL,
	UNIQUE (scan_type_id, name),
	FOREIGN KEY (scan_type_id) REFERENCES scanType (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS vulnerabilityCategory_scan_type_id ON vulnerabilityCategory (scan_type_id);

CREATE TABLE IF NOT EXISTS vulnerability (
	id SERIAL NOT NULL PRIMARY KEY,
	scan_id INTEGER NOT NULL,
	severity SEVERITY_TYPE DEFAULT 'medium'::SEVERITY_TYPE,
	status STATUS_TYPE DEFAULT 'not_reviewed'::STATUS_TYPE,
	category_id INTEGER,
	key_properties JSON,		-- ключевые свойства уязвимости, участвующие в дедупликации
	display_properties JSON,	-- свойства уязвимости, которые нужны только для отображения в UI
	first_found_at TIMESTAMP,	-- время, когда уязвимость была первый раз найдена
	tracker_ticket VARCHAR(128) NOT NULL DEFAULT '',
	FOREIGN KEY (scan_id) REFERENCES scan (id) ON DELETE CASCADE,
	FOREIGN KEY (category_id) REFERENCES vulnerabilityCategory (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS vulnerability_scan_id ON vulnerability (scan_id);
CREATE INDEX IF NOT EXISTS vulnerability_category_id ON vulnerability (category_id);
CREATE INDEX IF NOT EXISTS vulnerability_status ON vulnerability (status);
CREATE INDEX IF NOT EXISTS vulnerability_severity ON vulnerability (severity);

CREATE TABLE IF NOT EXISTS vulnerability2scanInstance (
	vulnerability_id INTEGER,
	scan_instance_id INTEGER NOT NULL,
	severity SEVERITY_TYPE DEFAULT 'medium'::SEVERITY_TYPE,
	category_id INTEGER,
	key_properties JSON,
	display_properties JSON,
	PRIMARY KEY (vulnerability_id, scan_instance_id),
	FOREIGN KEY (vulnerability_id) REFERENCES vulnerability (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_instance_id) REFERENCES scanInstance (id) ON DELETE CASCADE,
	FOREIGN KEY (category_id) REFERENCES vulnerabilityCategory (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS vulnerability2scanInstance_category_id ON vulnerability2scanInstance (category_id);

CREATE TABLE IF NOT EXISTS idmUser (
    id SERIAL NOT NULL PRIMARY KEY,
    login VARCHAR(64) UNIQUE NOT NULL,
    is_tvm_app BOOL DEFAULT 'f'
);

CREATE TABLE IF NOT EXISTS idmUserRole (
    user_id INTEGER NOT NULL,
    organization_id INTEGER NOT NULL,
    organization_slug VARCHAR(64) NOT NULL,
    project_id INTEGER NOT NULL,
    project_slug VARCHAR(64) NOT NULL,
    role VARCHAR(64) NOT NULL,
    PRIMARY KEY (user_id, organization_slug, project_slug, role),
    FOREIGN KEY (user_id) REFERENCES idmUser (id) ON DELETE CASCADE
);

DO
$$
BEGIN
	IF NOT EXISTS (SELECT * FROM pg_type typ INNER JOIN pg_namespace nsp ON nsp.oid = typ.typnamespace
					WHERE nsp.nspname = current_schema() AND typ.typname = 'history_action_type')
	THEN
		CREATE TYPE HISTORY_ACTION_TYPE AS ENUM ('change_status', 'add_comment');
	END IF;
END;
$$
LANGUAGE plpgsql;

CREATE TABLE IF NOT EXISTS historyAction (
	id SERIAL NOT NULL PRIMARY KEY,
	vulnerability_id INTEGER NOT NULL,
	action_type HISTORY_ACTION_TYPE DEFAULT 'add_comment'::HISTORY_ACTION_TYPE,
	action_time TIMESTAMP,
	action_text VARCHAR(1024) NOT NULL DEFAULT '',
	login VARCHAR(64),
	FOREIGN KEY (vulnerability_id) REFERENCES vulnerability (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS history_action_vulnerability_id ON historyAction (vulnerability_id);

CREATE TABLE IF NOT EXISTS codeQLDatabaseArchive (
    id SERIAL NOT NULL PRIMARY KEY,
    project_id INTEGER NOT NULL,
    language VARCHAR(16) NOT NULL DEFAULT '',
    tag VARCHAR(16) NOT NULL DEFAULT 'latest',
    mds_url VARCHAR(128) UNIQUE NOT NULL,
    archive_time TIMESTAMP,
    revision VARCHAR(64),
    FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS codeql_database_archive_project_id ON codeQLDatabaseArchive (project_id);

CREATE TABLE IF NOT EXISTS codeQLBuildTask (
    id SERIAL NOT NULL PRIMARY KEY,
    project_id INTEGER NOT NULL,
    sandbox_task_id INTEGER,
    start_time INTEGER,
    FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE
);
CREATE INDEX IF NOT EXISTS codeql_build_task_project_id ON codeQLBuildTask (project_id);

CREATE TABLE IF NOT EXISTS LastScanInstance (
	organization_id INTEGER NOT NULL,
	project_id INTEGER NOT NULL,
	scan_type_id INTEGER NOT NULL,
	scan_id INTEGER NOT NULL,
	scan_instance_id INTEGER NOT NULL,
	PRIMARY KEY (organization_id, project_id, scan_type_id, scan_id),
	FOREIGN KEY (organization_id) REFERENCES organization (id) ON DELETE CASCADE,
	FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_type_id) REFERENCES ScanType (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_id) REFERENCES scan (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_instance_id) REFERENCES ScanInstance (id) ON DELETE CASCADE
);

CREATE TABLE IF NOT EXISTS VulnerabilityTotalStatistics (
    organization_id INTEGER NOT NULL,
    project_id INTEGER NOT NULL,
	scan_type_id INTEGER NOT NULL,
    scan_id INTEGER NOT NULL,
    category_id INTEGER NOT NULL,
    total_vulnerabilities_count INTEGER NOT NULL,
    total_not_false_count INTEGER NOT NULL,
    total_not_reviewed_count INTEGER NOT NULL,
    blocker_not_false_count INTEGER NOT NULL,
    blocker_not_reviewed_count INTEGER NOT NULL,
    critical_not_false_count INTEGER NOT NULL,
    critical_not_reviewed_count INTEGER NOT NULL,
    medium_not_false_count INTEGER NOT NULL,
    medium_not_reviewed_count INTEGER NOT NULL,
    low_not_false_count INTEGER NOT NULL,
    low_not_reviewed_count INTEGER NOT NULL,
    info_not_false_count INTEGER NOT NULL,
    info_not_reviewed_count INTEGER NOT NULL,
    PRIMARY KEY (organization_id, project_id, scan_type_id, scan_id, category_id),
    FOREIGN KEY (organization_id) REFERENCES organization (id) ON DELETE CASCADE,
	FOREIGN KEY (project_id) REFERENCES project (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_type_id) REFERENCES ScanType (id) ON DELETE CASCADE,
	FOREIGN KEY (scan_id) REFERENCES scan (id) ON DELETE CASCADE,
	FOREIGN KEY (category_id) REFERENCES vulnerabilityCategory (id) ON DELETE CASCADE
);

CREATE OR REPLACE VIEW datalensVulnerabilityView
AS (SELECT v.id as vulnerability_id, o.name as organization_name, p.name project_name,
	st.title as scan_name, vc.name as vulnerability_category_name,
	v.severity, v.status, v.first_found_at
	FROM vulnerability as v
	JOIN vulnerabilitycategory as vc ON vc.id = v.category_id
	JOIN scan as s ON v.scan_id = s.id
	JOIN scantype as st ON s.scan_type_id = st.id
	JOIN project as p ON s.project_id = p.id
	JOIN organization as o ON p.organization_id = o.id
);

CREATE OR REPLACE VIEW datalensLastVulnerabilitiesView AS (
	SELECT v.id as vulnerability_id, o.name as organization_name, p.name project_name,
		st.title as scan_name, vc.name as vulnerability_category_name,
		v.severity, v.status, v.first_found_at
	FROM vulnerability as v
	JOIN vulnerability2scanInstance as v2si ON v2si.vulnerability_id = v.id
	JOIN vulnerabilitycategory as vc ON vc.id = v.category_id
	JOIN scan as s ON v.scan_id = s.id
	JOIN scantype as st ON s.scan_type_id = st.id
	JOIN project as p ON s.project_id = p.id
	JOIN organization as o ON p.organization_id = o.id
	WHERE v2si.scan_instance_id IN (
		SELECT MAX(si.id) as last_scan_instance_id
		FROM scanInstance si
		JOIN scan s ON si.scan_id = s.id
		JOIN project p ON p.id = s.project_id
		JOIN organization o ON o.id = p.organization_id
		GROUP BY o.id, p.id, si.scan_id
		)
);
`
	_, err := db.PG.Exec(sql)
	return err
}
