package insights

import (
	"context"

	"github.com/go-resty/resty/v2"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/impulse/models"
)

type (
	insightsAPI struct {
		Client    *resty.Client
		TVMClient tvm.Client
		DstTvmID  tvm.ClientID
	}

	Options struct {
		DstTvmID tvm.ClientID
		Endpoint string
	}
)

type API interface {
	SendMetrics(ctx context.Context, metrics []models.MetricForInsights) error
	SetTVMClient(tvmClient tvm.Client)
}

func New(opts *Options) *insightsAPI {
	httpc := resty.New().
		SetHeader("Content-Type", "application/json; charset=utf-8").
		SetBaseURL(opts.Endpoint)

	return &insightsAPI{
		Client:   httpc,
		DstTvmID: opts.DstTvmID,
	}
}

func (api *insightsAPI) SetTVMClient(tvmClient tvm.Client) {
	api.TVMClient = tvmClient
}

func (api *insightsAPI) SendMetrics(ctx context.Context, metrics []models.MetricForInsights) error {
	var ticket string
	var err error
	if api.TVMClient != nil {
		ticket, err = api.TVMClient.GetServiceTicketForID(ctx, api.DstTvmID)
		if err != nil {
			return err
		}
	}

	httpResp, err := api.Client.R().
		SetHeader("X-Ya-Service-Ticket", ticket).
		SetBody(&metrics).
		Post("/metrics")
	if err != nil {
		return err
	}
	if !httpResp.IsSuccess() {
		return xerrors.Errorf("Callback wrong status code from Insights %s", httpResp.Status())
	}

	return nil
}
