package auth

import (
	"fmt"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/api/internal/user"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_idmRepo "a.yandex-team.ru/security/impulse/api/repositories/idm"
	_idmUsecase "a.yandex-team.ru/security/impulse/api/usecases/idm"
)

type (
	AuthMiddleware struct {
		DB         *db.DB
		TVM        *tvmtool.Client
		BlackBox   blackbox.Client
		UseAuth    bool
		idmUsecase _idmUsecase.Usecase
	}
)

func NewAuthMiddleware(db *db.DB, tvm *tvmtool.Client, blackbox blackbox.Client, useAuth bool) *AuthMiddleware {
	idmRepo := _idmRepo.NewIdmRepository(db)
	idmUsecase := _idmUsecase.NewIdmUsecase(idmRepo)

	return &AuthMiddleware{
		DB:         db,
		TVM:        tvm,
		BlackBox:   blackbox,
		UseAuth:    useAuth,
		idmUsecase: idmUsecase,
	}
}

func (m *AuthMiddleware) WithAuth(next echo.HandlerFunc) echo.HandlerFunc {
	return func(e echo.Context) error {
		e.Set("useAuth", m.UseAuth)
		if !m.UseAuth {
			e.Set("user", &user.User{
				Login:    "current-user",
				IsTvmApp: false,
				ACL:      map[int]map[int]int{},
			})
			return next(e)
		}

		var bbUser *blackbox.User
		var err error
		var tvpApp string
		isTvmApp := false

		switch {
		case e.Request().Header.Get("Authorization") != "":
			bbUser, err = m.getOAuthUser(e)
		case e.Request().Header.Get("X-Ya-Service-Ticket") != "":
			isTvmApp = true
			tvpApp, err = m.getTvmService(e)
		default:
			bbUser, err = m.getCookieUser(e)
		}

		if err != nil {
			return utils.APIError(e, err)
		}

		var login string
		if isTvmApp {
			login = tvpApp
		} else {
			login = bbUser.Login
		}

		roles, err := m.idmUsecase.ListUserRolesByLogin(e.Request().Context(), login)
		if err != nil {
			return utils.APIError(e, err)
		}
		if len(roles) == 0 {
			return utils.APIForbidden(e)
		}

		acl := make(map[int]map[int]int)
		for _, role := range roles {
			_, ok := acl[role.OrganizationID]
			if !ok {
				acl[role.OrganizationID] = make(map[int]int)
			}
			prevRole, ok := acl[role.OrganizationID][role.ProjectID]
			if !ok || prevRole < user.RoleToInt(role.Role) {
				acl[role.OrganizationID][role.ProjectID] = user.RoleToInt(role.Role)
			}
		}

		impulseUser := &user.User{
			Login:    login,
			IsTvmApp: isTvmApp,
			ACL:      acl,
		}
		e.Set("user", impulseUser)

		return next(e)
	}
}

func (m *AuthMiddleware) getCookieUser(e echo.Context) (*blackbox.User, error) {
	sessID, err := e.Cookie("Session_id")
	if err != nil {
		return nil, xerrors.New("no session")
	}

	rsp, err := m.BlackBox.SessionID(
		e.Request().Context(),
		blackbox.SessionIDRequest{
			SessionID: sessID.Value,
			UserIP:    e.RealIP(),
			Host:      e.Request().Host,
		})

	if err != nil {
		return nil, xerrors.Errorf("failed to check user session: %w", err)
	}

	return &rsp.User, nil
}

func (m *AuthMiddleware) getOAuthUser(e echo.Context) (*blackbox.User, error) {
	token := e.Request().Header.Get("Authorization")
	if len(token) <= 6 || token[:6] != "OAuth " {
		return nil, xerrors.New("failed to parse OAuth token")
	}

	rsp, err := m.BlackBox.OAuth(
		e.Request().Context(),
		blackbox.OAuthRequest{
			OAuthToken: token[6:],
			UserIP:     e.RealIP(),
			Scopes:     []string{"impulse:all"},
		})

	if err != nil {
		return nil, xerrors.Errorf("failed to check OAuth token: %w", err)
	}

	return &rsp.User, nil
}

func (m *AuthMiddleware) getYandexUID(e echo.Context) string {
	var yandexUID string
	if cYandexUID, err := e.Cookie("yandexuid"); err == nil {
		yandexUID = cYandexUID.Value
	}
	return yandexUID
}

func (m *AuthMiddleware) getTvmService(e echo.Context) (string, error) {
	serviceTicket := e.Request().Header.Get("X-Ya-Service-Ticket")

	ticketInfo, err := m.TVM.CheckServiceTicket(e.Request().Context(), serviceTicket)
	if err != nil {
		return "", xerrors.Errorf("failed to check TVM ticket: %w", err)
	} else {
		return fmt.Sprint(ticketInfo.SrcID), nil
	}
}
