package sandbox

import (
	"fmt"
	"strings"

	"github.com/go-resty/resty/v2"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/security/impulse/models"
)

type apiImpl struct {
	Client *resty.Client
}

func New(opts *Options) *apiImpl {
	httpc := resty.New().
		SetHeader("Content-Type", "application/json; charset=utf-8").
		SetHeader("Authorization", fmt.Sprintf("OAuth %s", opts.OAuthToken)).
		SetBaseURL(opts.Endpoint)

	return &apiImpl{
		Client: httpc,
	}
}

func (api *apiImpl) Close() error {
	return nil
}

func (api *apiImpl) CreateTask(taskMessage *models.TaskMessageDTO, needDNS64 bool) (int, error) {
	description := strings.Join(taskMessage.Analysers, ", ")
	description = "imPulse SAST Workflow (" + description + ")"
	dns := "default"
	if needDNS64 {
		dns = "dns64"
	}
	request := CreateTaskRequest{
		Description: description,
		Owner:       "APPSEC_PIPELINE",
		Type:        "SAST_WORKFLOW",
		Priority: TaskPriority{
			Class:    "SERVICE",
			Subclass: "NORMAL",
		},
		Requirements: TaskRequirements{
			DNS: dns,
		},
	}
	request.CustomFields = []TaskCustomFields{
		{Name: "organization_id", Value: taskMessage.OrganizationID},
		{Name: "project_id", Value: taskMessage.ProjectID},
		{Name: "task_id", Value: taskMessage.TaskID},
		{Name: "analysers", Value: taskMessage.Analysers},
		{Name: "parameters", Value: taskMessage.Parameters},
	}

	var resp map[string]interface{}
	httpResp, err := api.Client.R().
		SetResult(&resp).
		SetBody(&request).
		Post("/api/v1.0/task")

	if err != nil {
		return 0, err
	}

	if !httpResp.IsSuccess() {
		if httpResp.StatusCode() == 429 {
			return 0, TooManyRequests{}
		}
		return 0, xerrors.Errorf("wrong status code for /task: %s", httpResp.Status())
	}

	if id, ok := resp["id"]; !ok {
		return 0, xerrors.Errorf("no Sandbox task ID in response")
	} else {
		return int(id.(float64)), nil
	}
}

func (api *apiImpl) CreateBuildCodeQLTask(organizationID, projectID int, repositoryURL string) (int, error) {
	description := "imPulse Build CodeQL"
	request := CreateTaskRequest{
		Description: description,
		Owner:       "APPSEC_PIPELINE",
		Type:        "BUILD_CODEQL_INDEX",
		Priority: TaskPriority{
			Class:    "SERVICE",
			Subclass: "NORMAL",
		},
		Requirements: TaskRequirements{
			DNS: "dns64",
		},
	}
	request.CustomFields = []TaskCustomFields{
		{Name: "organization_id", Value: organizationID},
		{Name: "project_id", Value: projectID},
		{Name: "repo_url", Value: repositoryURL},
	}

	var resp map[string]interface{}
	httpResp, err := api.Client.R().
		SetResult(&resp).
		SetBody(&request).
		Post("/api/v1.0/task")

	if err != nil {
		return 0, err
	}

	if !httpResp.IsSuccess() {
		if httpResp.StatusCode() == 429 {
			return 0, TooManyRequests{}
		}
		return 0, xerrors.Errorf("wrong status code for /task: %s", httpResp.Status())
	}

	if id, ok := resp["id"]; !ok {
		return 0, xerrors.Errorf("no Sandbox task ID in response")
	} else {
		return int(id.(float64)), nil
	}
}

func (api *apiImpl) StartTask(sandboxTaskID int) error {
	request := StartTaskRequest{ID: []int{sandboxTaskID}}
	var resp []StartTaskResponse
	httpResp, err := api.Client.R().
		SetResult(&resp).
		SetBody(&request).
		Put("/api/v1.0/batch/tasks/start")

	if err != nil {
		return err
	}

	if !httpResp.IsSuccess() {
		if httpResp.StatusCode() == 429 {
			return TooManyRequests{}
		}
		return xerrors.Errorf("wrong status code for /batch/tasks/start: %s", httpResp.Status())
	}

	if len(resp) == 0 {
		return xerrors.Errorf("empty response for /batch/tasks/start")
	}

	status := resp[0].Status

	if status != "SUCCESS" {
		return xerrors.Errorf("wrong status for Sandbox task %d: %s", sandboxTaskID, status)
	}

	return nil
}
