package user

import (
	"fmt"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/models"
)

type (
	User struct {
		Login    string
		IsTvmApp bool
		ACL      map[int]map[int]int
	}
)

var roleToInt = map[string]int{
	models.VIEW:   0,
	models.TRIAGE: 1,
	models.ADMIN:  2,
}

func RoleToInt(role string) int {
	return roleToInt[role]
}

func HasAccessToOrganization(e echo.Context, org int, minRole string) bool {
	if !e.Get("useAuth").(bool) {
		return true
	}

	currentUser := e.Get("user").(*User)

	role, ok := currentUser.ACL[0][0]
	if ok && role >= RoleToInt(minRole) {
		return true
	}
	role, ok = currentUser.ACL[org][0]
	if ok && role >= RoleToInt(minRole) {
		return true
	}

	return false
}

func HasAccessToAnyProject(e echo.Context, org int) bool {
	if !e.Get("useAuth").(bool) {
		return true
	}

	currentUser := e.Get("user").(*User)

	if _, ok := currentUser.ACL[0]; ok {
		return true
	}
	if _, ok := currentUser.ACL[org]; ok {
		return true
	}

	return false
}

func HasAccessToProject(e echo.Context, org, proj int, minRole string) bool {
	if !e.Get("useAuth").(bool) {
		return true
	}

	currentUser := e.Get("user").(*User)

	role, ok := currentUser.ACL[0][0]
	if ok && role >= RoleToInt(minRole) {
		return true
	}
	role, ok = currentUser.ACL[org][0]
	if ok && role >= RoleToInt(minRole) {
		return true
	}
	role, ok = currentUser.ACL[org][proj]
	if ok && role >= RoleToInt(minRole) {
		return true
	}

	return false
}

func UserInfo(e echo.Context) (*models.UserResponseDTO, error) {
	if !e.Get("useAuth").(bool) {
		return nil, xerrors.New("no authentication data")
	}

	currentUser := e.Get("user").(*User)

	if currentUser.IsTvmApp {
		return nil, xerrors.New("current user is TVM app")
	}

	return &models.UserResponseDTO{
		Login:     currentUser.Login,
		AvatarURL: fmt.Sprintf("https://center.yandex-team.ru/api/v1/user/%s/avatar/100.jpg", currentUser.Login),
	}, nil
}
