package utils

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/libs/go/simplelog"
)

// API response
// swagger:response apiResponse
type apiResponse struct {
	Ok     bool        `json:"ok"`
	Result interface{} `json:"result,omitempty"`
	Error  string      `json:"error,omitempty"`
}

func APIOk(c echo.Context, result interface{}) error {
	return c.JSON(http.StatusOK, apiResponse{
		Ok:     true,
		Result: result,
	})
}

func APIError(c echo.Context, err error) error {
	simplelog.Error("failed to process API request", "url", c.Request().URL, "err", err)

	return c.JSON(http.StatusInternalServerError, apiResponse{
		Ok:    false,
		Error: err.Error(),
	})
}

func APINotFound(c echo.Context) error {
	return c.JSON(http.StatusNotFound, apiResponse{
		Ok:    false,
		Error: "Not Found",
	})
}

func APIForbidden(c echo.Context) error {
	return c.JSON(http.StatusForbidden, apiResponse{
		Ok:    false,
		Error: "Go Away",
	})
}

func HTTPErrorHandler(err error, c echo.Context) {
	simplelog.Warn("request error", "uri", c.Request().URL, "err", err)

	var (
		code = http.StatusInternalServerError
		msg  interface{}
	)

	if he, ok := err.(*echo.HTTPError); ok {
		code = he.Code
		msg = he.Message
		if he.Internal != nil {
			msg = fmt.Errorf("%v, %v", err, he.Internal)
		}
	} else {
		msg = err.Error()
	}

	if _, ok := msg.(string); ok {
		msg = echo.Map{"error": msg}
	} else {
		msg = echo.Map{"error": "unknown"}
	}

	// Send response
	if !c.Response().Committed {
		if c.Request().Method == http.MethodHead { // Issue #608
			_ = c.NoContent(code)
		} else {
			_ = c.JSON(code, msg)
		}
	}
}
