package codeqldb

import (
	"context"
	"database/sql"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type codeQLDatabaseRepository struct {
	db *db.DB
}

func NewCodeQLDatabaseRepository(db *db.DB) Repository {
	return &codeQLDatabaseRepository{db}
}

func (c codeQLDatabaseRepository) Create(ctx context.Context,
	codeQLDatabase *models.CodeQLDatabase) (*models.CodeQLDatabase, error) {
	createdCodeQLDatabase := *codeQLDatabase
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := c.db.PG.QueryRowContext(ctx,
			"INSERT INTO codeQLDatabaseArchive (project_id, language, tag, mds_url, "+
				" 	archive_time, revision) "+
				" VALUES ($1, $2, $3, $4, $5, $6) "+
				" RETURNING id ", createdCodeQLDatabase.ProjectID, createdCodeQLDatabase.Language,
			createdCodeQLDatabase.Tag, createdCodeQLDatabase.MdsURL, createdCodeQLDatabase.ArchiveTime,
			createdCodeQLDatabase.Revision).Scan(&createdCodeQLDatabase.ID)
		if err != nil {
			return err
		}

		return nil
	})
	return &createdCodeQLDatabase, err
}

func (c codeQLDatabaseRepository) Update(ctx context.Context, codeQLDatabase *models.CodeQLDatabase) error {
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := c.db.PG.ExecContext(ctx,
			"UPDATE codeQLDatabaseArchive "+
				" SET project_id = $1, language = $2, tag = $3, mds_url = $4, "+
				" 	archive_time = NOW(), revision = $5 "+
				" WHERE id = $6", codeQLDatabase.ProjectID, codeQLDatabase.Language, codeQLDatabase.Tag,
			codeQLDatabase.MdsURL, codeQLDatabase.Revision, codeQLDatabase.ID)
		if err != nil {
			return err
		}
		return nil
	})
	return err
}

func (c codeQLDatabaseRepository) Get(ctx context.Context, projectID int, language string,
	tag string) (*models.CodeQLDatabase, error) {
	codeQLDatabase := models.CodeQLDatabase{}
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := c.db.PG.GetContext(ctx, &codeQLDatabase,
			"SELECT codeqldb.id as id, p.organization_id as organization_id, "+
				" 	codeqldb.project_id as project_id, codeqldb.language as language, "+
				" 	codeqldb.tag as tag, codeqldb.mds_url as mds_url, "+
				" 	codeqldb.archive_time as archive_time, codeqldb.revision as revision "+
				" FROM codeQLDatabaseArchive codeqldb "+
				" JOIN project p ON codeqldb.project_id = p.id "+
				" WHERE codeqldb.project_id = $1 AND codeqldb.language = $2 "+
				" 		AND codeqldb.tag = $3 ", projectID, language, tag)
		return err
	})
	if err != nil {
		return nil, err
	}
	return &codeQLDatabase, nil
}

func (c codeQLDatabaseRepository) ListTags(ctx context.Context, projectID int, language string) ([]string, error) {
	tags := make([]string, 0)
	var rows *sql.Rows
	err := c.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		rows, err = c.db.PG.QueryContext(ctx,
			"SELECT tag "+
				" FROM codeQLDatabaseArchive "+
				" WHERE project_id = $1 AND language = $2 "+
				" ORDER BY tag ASC", projectID, language)

		if err != nil {
			return err
		}

		defer func() {
			_ = rows.Close()
		}()

		tags = tags[:0]
		for rows.Next() {
			var tag string
			err = rows.Scan(&tag)
			if err != nil {
				return err
			}
			tags = append(tags, tag)
		}

		return rows.Err()
	})

	return tags, err
}

func (c codeQLDatabaseRepository) GetByMdsURL(ctx context.Context, mdsURL string) (*models.CodeQLDatabase, error) {
	codeQLDatabase := models.CodeQLDatabase{}
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := c.db.PG.GetContext(ctx, &codeQLDatabase,
			"SELECT codeqldb.id as id, p.organization_id as organization_id, "+
				" 	codeqldb.project_id as project_id, codeqldb.language as language, "+
				" 	codeqldb.tag as tag, codeqldb.mds_url as mds_url, "+
				" 	codeqldb.archive_time as archive_time, codeqldb.revision as revision "+
				" FROM codeQLDatabaseArchive codeqldb "+
				" JOIN project p ON codeqldb.project_id = p.id "+
				" WHERE codeqldb.mds_url = $1 ", mdsURL)
		return err
	})
	if err != nil {
		return nil, err
	}
	return &codeQLDatabase, nil
}

func (c codeQLDatabaseRepository) CreateBuildTask(ctx context.Context,
	task *models.CodeQLBuildTask) (*models.CodeQLBuildTask, error) {
	createdTask := *task
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := c.db.PG.QueryRowContext(ctx,
			"INSERT INTO codeQLBuildTask (project_id, sandbox_task_id, start_time) "+
				" VALUES ($1, $2, $3) "+
				" RETURNING id ", task.ProjectID, task.SandboxTaskID, task.StartTime).Scan(&createdTask.ID)
		if err != nil {
			return err
		}

		return nil
	})
	return &createdTask, err
}

func (c codeQLDatabaseRepository) GetOldestBuildTasks(ctx context.Context,
	timestamp int64, count int) ([]*models.CodeQLBuildOldestTask, error) {
	tasks := make([]*models.CodeQLBuildOldestTask, 0)
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		return c.db.PG.SelectContext(ctx, &tasks,
			"SELECT * FROM ( "+
				"SELECT p.id as project_id, o.id as organization_id, MAX(COALESCE(c.start_time, 0)) as start_time "+
				"FROM project p left join codeQLBuildTask c on p.id = c.project_id, organization o "+
				"WHERE p.organization_id = o.id and p.build_codeql_index = true "+
				"GROUP BY p.id, o.id ORDER BY start_time, project_id) AS q "+
				"WHERE start_time < $1 LIMIT $2",
			timestamp, count)
	})
	if err != nil {
		return nil, err
	}
	return tasks, nil
}

func (c codeQLDatabaseRepository) Delete(ctx context.Context,
	codeQLDatabase models.CodeQLDatabase) error {
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := c.db.PG.ExecContext(ctx,
			"DELETE FROM codeQLDatabaseArchive WHERE id = $1", codeQLDatabase.ID)
		if err != nil {
			return err
		}
		return nil
	})
	return err
}
