package cron

import (
	"context"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type cronRepository struct {
	db *db.DB
}

func NewCronRepository(db *db.DB) Repository {
	return &cronRepository{db}
}

func (c cronRepository) Create(ctx context.Context, cron *models.Cron) (int, error) {
	createdCron := *cron
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := c.db.PG.QueryRowContext(ctx, `
INSERT INTO
	cron (spec, next_time, description, is_running, organization_id, project_id, workflow_id, parameters, analysers, callback_url, non_template_scan)
VALUES
	($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
RETURNING id`,
			cron.Spec, cron.NextTime, cron.Description, cron.IsRunning, cron.OrganizationID, cron.ProjectID, cron.WorkflowID, cron.Parameters, cron.Analysers, cron.CallbackURL, cron.NonTemplateScan).Scan(&createdCron.ID)
		if err != nil {
			return err
		}

		return nil
	})
	return createdCron.ID, err
}

func (c cronRepository) List(ctx context.Context, organizationID, projectID int) (t []*models.CronResponseDTO, err error) {
	t = make([]*models.CronResponseDTO, 0)
	err = c.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		err = c.db.PG.SelectContext(ctx, &t, `
SELECT
	c.id as id, c.spec as spec, c.next_time as next_time, c.description as description, c.is_running as is_running,
	c.organization_id as organization_id, c.project_id as project_id,
	c.workflow_id as workflow_id, c.parameters as parameters, c.analysers as analysers, c.callback_url as callback_url,
	c.non_template_scan as non_template_scan, COALESCE(w.name, '') as workflow
FROM
	cron c
LEFT JOIN
	workflow w
ON
	c.workflow_id = w.id
WHERE
	organization_id = $1 AND project_id = $2
ORDER BY
	description`, organizationID, projectID)

		return
	})
	return
}

func (c cronRepository) ListAll(ctx context.Context) (t []*models.Cron, err error) {
	t = make([]*models.Cron, 0)
	err = c.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		err = c.db.PG.SelectContext(ctx, &t, `
SELECT
	*
FROM
	cron`)

		return
	})
	return
}

func (c cronRepository) GetByID(ctx context.Context, organizationID, projectID, id int) (*models.Cron, error) {
	cronTask := models.Cron{}
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := c.db.PG.GetContext(ctx, &cronTask, `
SELECT
	*
FROM
	cron
WHERE
	id = $1 AND organization_id = $2 AND project_id = $3`, id, organizationID, projectID)
		return err
	})
	if err != nil {
		return nil, err
	}
	return &cronTask, err
}

func (c cronRepository) Update(ctx context.Context, cron *models.Cron) error {
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := c.db.PG.ExecContext(ctx, `
UPDATE
	cron
SET
	parameters = $1, analysers = $2, spec = $3, next_time = $4, description = $5, callback_url = $6, non_template_scan = $7
WHERE
	id = $8 AND organization_id = $9 AND project_id = $10`,
			cron.Parameters, cron.Analysers, cron.Spec, cron.NextTime, cron.Description, cron.CallbackURL, cron.NonTemplateScan, cron.ID, cron.OrganizationID, cron.ProjectID)
		return err
	})
	return err
}

func (c cronRepository) DeleteByID(ctx context.Context, organizationID, projectID, id int) error {
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := c.db.PG.ExecContext(ctx, `DELETE FROM cron WHERE id = $1 AND organization_id = $2 AND project_id = $3`,
			id, organizationID, projectID)
		return err
	})
	return err
}

func (c cronRepository) UpdateRunningState(ctx context.Context, organizationID, projectID, id int, isRunning bool) error {
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := c.db.PG.ExecContext(ctx, `UPDATE cron SET is_running = $1 WHERE id = $2 AND organization_id = $3 AND project_id = $4`,
			isRunning, id, organizationID, projectID)
		return err
	})
	return err
}

func (c cronRepository) UpdateNextTime(ctx context.Context, organizationID, projectID, id int, nextTime int64) error {
	err := c.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := c.db.PG.ExecContext(ctx, `UPDATE cron SET next_time = $1 WHERE id = $2 AND organization_id = $3 AND project_id = $4`,
			nextTime, id, organizationID, projectID)
		return err
	})
	return err
}
