package idm

import (
	"context"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type idmRepository struct {
	db *db.DB
}

func NewIdmRepository(db *db.DB) Repository {
	return &idmRepository{db}
}

func (i idmRepository) Create(ctx context.Context, user *models.IdmUser) (*models.IdmUser, error) {
	createdUser := *user
	err := i.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := i.db.PG.QueryRowContext(ctx, `INSERT INTO idmUser (login, is_tvm_app) VALUES ($1, $2) RETURNING id`,
			user.Login, user.IsTvmApp).Scan(&createdUser.ID)
		return err
	})
	return &createdUser, err
}

func (i idmRepository) GetByLogin(ctx context.Context, login string) (*models.IdmUser, error) {
	user := models.IdmUser{}
	err := i.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := i.db.PG.GetContext(ctx, &user, `SELECT id, login, is_tvm_app FROM idmUser WHERE login = $1`, login)
		return err
	})
	if err != nil {
		return nil, err
	}
	return &user, err
}

func (i idmRepository) RemoveByLogin(ctx context.Context, login string) error {
	err := i.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := i.db.PG.ExecContext(ctx, `DELETE FROM idmUser WHERE login = $1`, login)
		return err
	})
	return err
}

func (i idmRepository) AddUserRole(ctx context.Context, role *models.IdmUserRole) error {
	err := i.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := i.db.PG.NamedExecContext(ctx, `
INSERT INTO
	idmUserRole (user_id, organization_id, organization_slug, project_id, project_slug, role)
VALUES
	(:user_id, :organization_id, :organization_slug, :project_id, :project_slug, :role)
`, role)
		return err
	})
	return err
}

func (i idmRepository) RemoveUserRoleByLogin(ctx context.Context, login string, role *models.IdmRoleDTO) error {
	err := i.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := i.db.PG.ExecContext(ctx, `
DELETE FROM
	idmUserRole
WHERE
	user_id=ANY(
		SELECT id FROM idmUser
		WHERE login=$1)
	AND organization_slug = $2 AND project_slug = $3 AND role = $4
`, login, role.Organization, role.Project, role.Role)
		return err
	})
	return err
}

func (i idmRepository) ListUserRolesByLogin(ctx context.Context, login string) (r []*models.IdmUserRole, err error) {
	r = make([]*models.IdmUserRole, 0)
	err = i.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		err = i.db.PG.SelectContext(ctx, &r, `
SELECT
	idmUserRole.user_id as user_id, idmUserRole.organization_id as organization_id, idmUserRole.organization_slug as organization_slug,
	idmUserRole.project_id as project_id, idmUserRole.project_slug as project_slug, idmUserRole.role as role
FROM
	idmUserRole, idmUser
WHERE
	idmUserRole.user_id = idmUser.id AND idmUser.login = $1`, login)
		return
	})
	return
}

func (i idmRepository) ListRoles(ctx context.Context) (r []*models.IdmUserRole, err error) {
	r = make([]*models.IdmUserRole, 0)
	err = i.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		err = i.db.PG.SelectContext(ctx, &r, `SELECT * FROM idmUserRole`)
		return
	})
	return
}

func (i idmRepository) ListUsers(ctx context.Context) (u []*models.IdmUser, err error) {
	u = make([]*models.IdmUser, 0)
	err = i.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		err = i.db.PG.SelectContext(ctx, &u, `SELECT * FROM idmUser`)
		return
	})
	return
}
