package organization

import (
	"context"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type organizationRepository struct {
	db *db.DB
}

func NewOrganizationRepository(db *db.DB) Repository {
	return &organizationRepository{db}
}

func (o organizationRepository) Create(ctx context.Context, org *models.Organization) (models.Organization, error) {
	createdOrg := *org
	err := o.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := o.db.PG.QueryRowContext(ctx,
			`INSERT INTO organization (name, slug, tracker_queue, abc_service_id) VALUES ($1, $2, $3, $4) RETURNING id`,
			org.Name, org.Slug, org.TrackerQueue, org.ABCServiceID).Scan(&createdOrg.ID)
		if err != nil {
			return err
		}

		return nil
	})
	return createdOrg, err
}

func (o organizationRepository) Update(ctx context.Context, org *models.Organization) error {
	err := o.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := o.db.PG.ExecContext(ctx,
			`UPDATE organization SET name = $1, tracker_queue = $2, abc_service_id = $3 WHERE id = $4`,
			org.Name, org.TrackerQueue, org.ABCServiceID, org.ID)
		if err != nil {
			return err
		}
		return nil
	})
	return err
}

func (o organizationRepository) GetByID(ctx context.Context, id int) (*models.Organization, error) {
	org := models.Organization{}
	err := o.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := o.db.PG.GetContext(ctx, &org,
			`SELECT id, name, slug, tracker_queue, abc_service_id FROM organization WHERE id = $1 AND deleted = FALSE`, id)
		return err
	})
	if err != nil {
		return nil, err
	}
	return &org, err
}

func (o organizationRepository) GetBySlug(ctx context.Context, slug string) (*models.Organization, error) {
	org := models.Organization{}
	err := o.db.Trier.Try(ctx, func(ctx context.Context) error {
		err := o.db.PG.GetContext(ctx, &org,
			`SELECT id, name, slug, tracker_queue, abc_service_id FROM organization WHERE slug = $1 AND deleted = FALSE`, slug)
		return err
	})
	if err != nil {
		return nil, err
	}
	return &org, err
}

func (o organizationRepository) List(ctx context.Context) ([]*models.OrganizationInfo, error) {
	organizations := make([]*models.OrganizationInfo, 0)
	err := o.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		rows, err := o.db.PG.QueryContext(ctx,
			"SELECT o.id AS id, o.name AS name, o.slug AS slug, "+
				" o.tracker_queue AS tracker_queue, o.abc_service_id AS abc_service_id, "+
				" COALESCE(SUM(vts.total_vulnerabilities_count),0) AS \"statistics.total_vulnerabilities_count\", "+
				" COALESCE(SUM(vts.total_not_false_count),0) as \"statistics.total_not_false_count\", "+
				" COALESCE(SUM(vts.total_not_reviewed_count),0) as \"statistics.total_not_reviewed_count\", "+
				" COALESCE(SUM(vts.blocker_not_false_count),0) as \"statistics.blocker_not_false_count\", "+
				" COALESCE(SUM(vts.blocker_not_reviewed_count),0) as \"statistics.blocker_not_reviewed_count\", "+
				" COALESCE(SUM(vts.critical_not_false_count),0) as \"statistics.critical_not_false_count\", "+
				" COALESCE(SUM(vts.critical_not_reviewed_count),0) as \"statistics.critical_not_reviewed_count\", "+
				" COALESCE(SUM(vts.medium_not_false_count),0) as \"statistics.medium_not_false_count\", "+
				" COALESCE(SUM(vts.medium_not_reviewed_count),0) as \"statistics.medium_not_reviewed_count\", "+
				" COALESCE(SUM(vts.low_not_false_count),0) as \"statistics.low_not_false_count\", "+
				" COALESCE(SUM(vts.low_not_reviewed_count),0) as \"statistics.low_not_reviewed_count\", "+
				" COALESCE(SUM(vts.info_not_false_count),0) as \"statistics.info_not_false_count\", "+
				" COALESCE(SUM(vts.info_not_reviewed_count),0) as \"statistics.info_not_reviewed_count\" "+
				" FROM organization AS o "+
				" LEFT JOIN vulnerabilityTotalStatistics AS vts ON o.id = vts.organization_id "+
				" WHERE o.deleted = FALSE "+
				" GROUP BY o.id, o.name, o.slug, o.tracker_queue, o.abc_service_id "+
				" ORDER BY o.name ASC")

		if err != nil {
			return err
		}

		defer func() {
			_ = rows.Close()
		}()

		organizations = organizations[:0]
		for rows.Next() {
			organization := new(models.OrganizationInfo)
			err = rows.Scan(&organization.ID, &organization.Name, &organization.Slug,
				&organization.TrackerQueue, &organization.ABCServiceID,
				&organization.Statistics.TotalVulnerabilitiesCount,
				&organization.Statistics.TotalNotFalseVulnerabilitiesCount,
				&organization.Statistics.TotalNotReviewedVulnerabilitiesCount,
				&organization.Statistics.BlockerNotFalseVulnerabilitiesCount,
				&organization.Statistics.BlockerNotReviewedVulnerabilitiesCount,
				&organization.Statistics.CriticalNotFalseVulnerabilitiesCount,
				&organization.Statistics.CriticalNotReviewedVulnerabilitiesCount,
				&organization.Statistics.MediumNotFalseVulnerabilitiesCount,
				&organization.Statistics.MediumNotReviewedVulnerabilitiesCount,
				&organization.Statistics.LowNotFalseVulnerabilitiesCount,
				&organization.Statistics.LowNotReviewedVulnerabilitiesCount,
				&organization.Statistics.InfoNotFalseVulnerabilitiesCount,
				&organization.Statistics.InfoNotReviewedVulnerabilitiesCount)
			if err != nil {
				return err
			}
			organizations = append(organizations, organization)
		}

		return rows.Err()
	})

	return organizations, err
}

func (o organizationRepository) DeleteByID(ctx context.Context, id int) error {
	err := o.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := o.db.PG.ExecContext(ctx, `UPDATE organization SET deleted = TRUE WHERE id = $1`, id)
		if err != nil {
			return err
		}
		return nil
	})
	return err
}

func (o organizationRepository) GetSummaryStatisticsByID(ctx context.Context,
	id int) (*models.OrganizationStatistics, error) {
	organizationStatistics := models.OrganizationStatistics{}
	err := o.db.Trier.Try(ctx, func(ctx context.Context) error {
		return o.db.PG.GetContext(ctx, &organizationStatistics,
			"SELECT COALESCE(SUM(total_vulnerabilities_count),0) as total_vulnerabilities_count, "+
				" COALESCE(SUM(total_not_false_count),0) as total_not_false_count, "+
				" COALESCE(SUM(total_not_reviewed_count),0) as total_not_reviewed_count, "+
				" COALESCE(SUM(blocker_not_false_count),0) as blocker_not_false_count, "+
				" COALESCE(SUM(blocker_not_reviewed_count),0) as blocker_not_reviewed_count, "+
				" COALESCE(SUM(critical_not_false_count),0) as critical_not_false_count, "+
				" COALESCE(SUM(critical_not_reviewed_count),0) as critical_not_reviewed_count, "+
				" COALESCE(SUM(medium_not_false_count),0) as medium_not_false_count, "+
				" COALESCE(SUM(medium_not_reviewed_count),0) as medium_not_reviewed_count, "+
				" COALESCE(SUM(low_not_false_count),0) as low_not_false_count, "+
				" COALESCE(SUM(low_not_reviewed_count),0) as low_not_reviewed_count, "+
				" COALESCE(SUM(info_not_false_count),0) as info_not_false_count, "+
				" COALESCE(SUM(info_not_reviewed_count),0) as info_not_reviewed_count "+
				" FROM vulnerabilityTotalStatistics "+
				" WHERE organization_id = $1 ",
			id)
	})
	if err != nil {
		return nil, err
	}
	return &organizationStatistics, nil
}
