package project

import (
	"context"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type projectRepository struct {
	db *db.DB
}

func (p projectRepository) Create(ctx context.Context, project *models.Project) (models.Project, error) {
	createdProject := *project
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		return p.db.PG.QueryRowContext(ctx, `INSERT INTO project (name, slug, organization_id, tracker_queue, `+
			`abc_service_id, tags, notification_settings, build_codeql_index) `+
			`VALUES ($1, $2, $3, $4, $5, $6, $7, $8) RETURNING id`, project.Name, project.Slug, project.OrganizationID,
			project.TrackerQueue, project.ABCServiceID, project.Tags, project.NotificationSettings,
			project.BuildCodeQLIndex).Scan(&createdProject.ID)
	})
	return createdProject, err
}

func (p projectRepository) Update(ctx context.Context, project *models.Project) error {
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := p.db.PG.ExecContext(ctx,
			`UPDATE project SET name = $1, organization_id = $2, tracker_queue = $3, abc_service_id = $4, `+
				`tags = $5, build_codeql_index = $6, notification_settings = $7 WHERE id = $8`,
			project.Name, project.OrganizationID, project.TrackerQueue, project.ABCServiceID,
			project.Tags, project.BuildCodeQLIndex, project.NotificationSettings, project.ID)
		return err
	})
	return err
}

func (p projectRepository) GetByID(ctx context.Context, id int) (*models.Project, error) {
	project := models.Project{}
	err := p.db.PG.GetContext(ctx, &project,
		" SELECT project.id as id, project.name as name, project.slug as slug, "+
			" project.organization_id as organization_id, project.tags as tags, project.notification_settings as notification_settings, "+
			" project.build_codeql_index as build_codeql_index, "+
			" CASE WHEN project.tracker_queue <> '' THEN project.tracker_queue ELSE organization.tracker_queue END as tracker_queue, "+
			" CASE WHEN project.abc_service_id <> 0 THEN project.abc_service_id ELSE organization.abc_service_id END as abc_service_id "+
			" FROM project, organization "+
			" WHERE project.organization_id = organization.id AND project.id = $1 "+
			" 	 AND project.deleted = FALSE AND organization.deleted = FALSE", id)
	if err != nil {
		return nil, err
	}
	return &project, nil
}

func (p projectRepository) GetBySlug(ctx context.Context, slug string) (*models.Project, error) {
	project := models.Project{}
	err := p.db.PG.GetContext(ctx, &project,
		" SELECT project.id as id, project.name as name, project.slug as slug, "+
			" project.organization_id as organization_id, project.tags as tags, project.notification_settings as notification_settings, "+
			" project.build_codeql_index as build_codeql_index, "+
			" CASE WHEN project.tracker_queue <> '' THEN project.tracker_queue ELSE organization.tracker_queue END as tracker_queue, "+
			" CASE WHEN project.abc_service_id <> 0 THEN project.abc_service_id ELSE organization.abc_service_id END as abc_service_id "+
			" FROM project, organization "+
			" WHERE project.organization_id = organization.id AND project.slug = $1 "+
			" 	AND project.deleted = FALSE AND organization.deleted = FALSE", slug)
	if err != nil {
		return nil, err
	}
	return &project, nil
}

func (p projectRepository) GetByOrganizationIDAndProjectID(ctx context.Context, organizationID int, projectID int) (*models.Project, error) {
	project := models.Project{}
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		return p.db.PG.GetContext(ctx, &project,
			" SELECT project.id as id, project.name as name, project.slug as slug, "+
				" project.organization_id as organization_id, project.tags as tags, project.notification_settings as notification_settings, "+
				" project.build_codeql_index as build_codeql_index, "+
				" CASE WHEN project.tracker_queue <> '' THEN project.tracker_queue ELSE organization.tracker_queue END as tracker_queue, "+
				" CASE WHEN project.abc_service_id <> 0 THEN project.abc_service_id ELSE organization.abc_service_id END as abc_service_id "+
				" FROM project, organization "+
				" WHERE project.organization_id = organization.id AND project.organization_id = $1 AND project.id = $2 "+
				" 	AND project.deleted = FALSE AND organization.deleted = FALSE ",
			organizationID, projectID)
	})
	if err != nil {
		return nil, err
	}
	return &project, nil
}

func (p projectRepository) ListByOrganizationID(ctx context.Context, organizationID int) ([]*models.ProjectInfo, error) {
	projects := make([]*models.ProjectInfo, 0)
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		return p.db.PG.SelectContext(ctx, &projects,
			"SELECT project.id as id, project.name as name, project.slug as slug, "+
				" 	project.organization_id as organization_id, project.tags as tags, project.notification_settings as notification_settings, "+
				"   project.build_codeql_index as build_codeql_index, "+
				" 	CASE WHEN project.tracker_queue <> '' THEN project.tracker_queue ELSE organization.tracker_queue END as tracker_queue, "+
				" 	CASE WHEN project.abc_service_id <> 0 THEN project.abc_service_id ELSE organization.abc_service_id END as abc_service_id, "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.total_vulnerabilities_count),0) as \"statistics.total_vulnerabilities_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.total_not_false_count),0) as \"statistics.total_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.total_not_reviewed_count),0) as \"statistics.total_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.blocker_not_false_count),0) as \"statistics.blocker_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.blocker_not_reviewed_count),0) as \"statistics.blocker_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.critical_not_false_count),0) as \"statistics.critical_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.critical_not_reviewed_count),0) as \"statistics.critical_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.medium_not_false_count),0) as \"statistics.medium_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.medium_not_reviewed_count),0) as \"statistics.medium_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.low_not_false_count),0) as \"statistics.low_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.low_not_reviewed_count),0) as \"statistics.low_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.info_not_false_count),0) as \"statistics.info_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.info_not_reviewed_count),0) as \"statistics.info_not_reviewed_count\" "+
				" FROM project "+
				" JOIN organization ON organization.id = project.organization_id "+
				" LEFT JOIN vulnerabilityTotalStatistics ON vulnerabilityTotalStatistics.project_id = project.id "+
				" WHERE project.organization_id = organization.id AND project.organization_id = $1 "+
				" 	AND project.deleted = FALSE"+
				" GROUP BY project.id, organization.tracker_queue, organization.abc_service_id "+
				" ORDER BY project.name ASC",
			organizationID)
	})
	if err != nil {
		return nil, err
	}
	return projects, nil
}

func (p projectRepository) ListWithEnabledNotifications(ctx context.Context) ([]*models.ProjectInfo, error) {
	projects := make([]*models.ProjectInfo, 0)
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		return p.db.PG.SelectContext(ctx, &projects,
			"SELECT project.id as id, project.name as name, project.slug as slug, "+
				" 	project.organization_id as organization_id, project.tags as tags, project.notification_settings as notification_settings, "+
				"   project.build_codeql_index as build_codeql_index, "+
				" 	CASE WHEN project.tracker_queue <> '' THEN project.tracker_queue ELSE organization.tracker_queue END as tracker_queue, "+
				" 	CASE WHEN project.abc_service_id <> 0 THEN project.abc_service_id ELSE organization.abc_service_id END as abc_service_id, "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.total_vulnerabilities_count),0) as \"statistics.total_vulnerabilities_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.total_not_false_count),0) as \"statistics.total_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.total_not_reviewed_count),0) as \"statistics.total_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.blocker_not_false_count),0) as \"statistics.blocker_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.blocker_not_reviewed_count),0) as \"statistics.blocker_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.critical_not_false_count),0) as \"statistics.critical_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.critical_not_reviewed_count),0) as \"statistics.critical_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.medium_not_false_count),0) as \"statistics.medium_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.medium_not_reviewed_count),0) as \"statistics.medium_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.low_not_false_count),0) as \"statistics.low_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.low_not_reviewed_count),0) as \"statistics.low_not_reviewed_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.info_not_false_count),0) as \"statistics.info_not_false_count\", "+
				" 	COALESCE(SUM(vulnerabilityTotalStatistics.info_not_reviewed_count),0) as \"statistics.info_not_reviewed_count\" "+
				" FROM project "+
				" JOIN organization ON organization.id = project.organization_id "+
				" LEFT JOIN vulnerabilityTotalStatistics ON vulnerabilityTotalStatistics.project_id = project.id "+
				" WHERE project.deleted = FALSE "+
				" 	AND (project.notification_settings::json->>'enabled')::boolean = 't' "+
				" GROUP BY project.id, organization.tracker_queue, organization.abc_service_id "+
				" ORDER BY project.name ASC")
	})
	if err != nil {
		return nil, err
	}
	return projects, nil
}

func (p projectRepository) DeleteByID(ctx context.Context, id int) error {
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := p.db.PG.ExecContext(ctx, `UPDATE project SET deleted = TRUE WHERE id = $1`, id)
		if err != nil {
			return err
		}
		return nil
	})
	return err
}

func (p projectRepository) ListTags(ctx context.Context) ([]string, error) {
	var tags []string
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		return p.db.PG.SelectContext(ctx, &tags,
			"select distinct unnest(tags) from project",
		)
	})
	if err != nil {
		return nil, err
	}
	return tags, nil
}

func (p projectRepository) GetStatisticsByID(ctx context.Context,
	id int) (*models.ProjectStatistics, error) {
	projectStatistics := models.ProjectStatistics{}
	err := p.db.Trier.Try(ctx, func(ctx context.Context) error {
		return p.db.PG.GetContext(ctx, &projectStatistics,
			"SELECT COALESCE(SUM(total_vulnerabilities_count),0) as total_vulnerabilities_count, "+
				" COALESCE(SUM(total_not_false_count),0) as total_not_false_count, "+
				" COALESCE(SUM(total_not_reviewed_count),0) as total_not_reviewed_count, "+
				" COALESCE(SUM(blocker_not_false_count),0) as blocker_not_false_count, "+
				" COALESCE(SUM(blocker_not_reviewed_count),0) as blocker_not_reviewed_count, "+
				" COALESCE(SUM(critical_not_false_count),0) as critical_not_false_count, "+
				" COALESCE(SUM(critical_not_reviewed_count),0) as critical_not_reviewed_count, "+
				" COALESCE(SUM(medium_not_false_count),0) as medium_not_false_count, "+
				" COALESCE(SUM(medium_not_reviewed_count),0) as medium_not_reviewed_count, "+
				" COALESCE(SUM(low_not_false_count),0) as low_not_false_count, "+
				" COALESCE(SUM(low_not_reviewed_count),0) as low_not_reviewed_count, "+
				" COALESCE(SUM(info_not_false_count),0) as info_not_false_count, "+
				" COALESCE(SUM(info_not_reviewed_count),0) as info_not_reviewed_count "+
				" FROM vulnerabilityTotalStatistics "+
				" WHERE project_id = $1 ",
			id)
	})
	if err != nil {
		return nil, err
	}
	return &projectStatistics, nil
}

func NewProjectRepository(db *db.DB) Repository {
	return &projectRepository{db: db}
}
