package scan

import (
	"context"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type scanRepository struct {
	db *db.DB
}

func NewScanRepository(db *db.DB) Repository {
	return &scanRepository{db}
}

func (s scanRepository) GetByID(ctx context.Context, id int) (*models.Scan, error) {
	scan := models.Scan{}
	err := s.db.PG.GetContext(ctx, &scan, "SELECT * FROM scan WHERE id = $1", id)
	if err != nil {
		return nil, err
	}
	return &scan, nil
}

func (s scanRepository) ListByProjectID(ctx context.Context, projectID int) ([]*models.Scan, error) {
	scans := make([]*models.Scan, 0)
	err := s.db.Trier.Try(ctx, func(ctx context.Context) error {
		return s.db.PG.SelectContext(ctx, &scans,
			"SELECT * FROM scan WHERE project_id=$1", projectID)
	})
	if err != nil {
		return nil, err
	}
	return scans, nil
}

func (s scanRepository) GetByProjectIDAndScanTypeName(ctx context.Context, projectID int, scanTypeName string) (scanPtr *models.Scan, err error) {
	scanPtr = new(models.Scan)
	err = s.db.PG.GetContext(ctx, scanPtr,
		"SELECT scan.id as id, scan.project_id as project_id, "+
			" scan.scan_type_id as scan_type_id, scan.last_update_token as last_update_token "+
			" FROM scan, scanType "+
			" WHERE scan.project_id = $1 AND scanType.id = scan.scan_type_id AND scanType.type_name = $2",
		projectID, scanTypeName)
	if err != nil {
		return nil, err
	}
	return
}

func (s scanRepository) CreateNew(ctx context.Context, project *models.Project, scanType *models.ScanType) (*models.Scan, error) {
	scan := new(models.Scan)
	newLastUpdateToken := uuid.Must(uuid.NewV4()).String()
	err := s.db.PG.QueryRowContext(ctx, "INSERT INTO scan (project_id, scan_type_id, last_update_token) "+
		" VALUES ($1, $2, $3) RETURNING id, project_id, scan_type_id, last_update_token",
		project.ID, scanType.ID, newLastUpdateToken).Scan(&scan.ID, &scan.ProjectID,
		&scan.ScanTypeID, &scan.LastUpdateToken)
	if err != nil {
		return nil, err
	}
	return scan, nil
}

func (s scanRepository) DeleteByID(ctx context.Context, id int) error {
	_, err := s.db.PG.ExecContext(ctx, "DELETE FROM scan WHERE id = $1", id)
	if err != nil {
		return err
	}
	return nil
}

func (s scanRepository) GetLastScanInstance(ctx context.Context, scan *models.Scan) (*models.ScanInstance, error) {
	scanInstance := models.ScanInstance{}
	err := s.db.PG.GetContext(ctx, &scanInstance,
		"SELECT * FROM scanInstance WHERE scan_id = $1 ORDER BY end_time DESC LIMIT 1", scan.ID)
	if err != nil {
		return nil, err
	}
	return &scanInstance, nil
}
