package vulnerabilityhistoryaction

import (
	"context"
	"fmt"
	"strings"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/models"
)

type vulnerabilityHistoryActionRepository struct {
	db *db.DB
}

func NewVulnerabilityHistoryActionRepository(db *db.DB) Repository {
	return &vulnerabilityHistoryActionRepository{db}
}

func (m *vulnerabilityHistoryActionRepository) FetchByVulnerabilityID(ctx context.Context, vulnerabilityID int) ([]*models.VulnerabilityHistoryAction, error) {
	historyActions := make([]*models.VulnerabilityHistoryAction, 0)
	err := m.db.Trier.Try(ctx, func(ctx context.Context) (err error) {
		err = m.db.PG.SelectContext(ctx, &historyActions, `
SELECT
	*
FROM
	historyAction
WHERE
	vulnerability_id = $1
ORDER BY action_time DESC`,
			vulnerabilityID)

		if err != nil {
			if !db.IsRetriableError(err) {
				return
			}

			err = xerrors.Errorf("failed to list vulnerability history actions: %w", err)
			return
		}
		return
	})
	return historyActions, err
}

func (m *vulnerabilityHistoryActionRepository) AddHistoryActions(ctx context.Context, vulnerabilityHistoryActions []models.VulnerabilityHistoryAction) error {
	values := []interface{}{}
	placeholders := []string{}
	for idx, action := range vulnerabilityHistoryActions {
		placeholders = append(placeholders, fmt.Sprintf("($%d, $%d, NOW(), $%d, $%d)", idx*4+1, idx*4+2, idx*4+3, idx*4+4))
		values = append(values, action.VulnerabilityID,
			action.ActionType, action.ActionText, action.Login)
	}
	query := "INSERT INTO historyAction (vulnerability_id, action_type, action_time, action_text, login)  VALUES " + strings.Join(placeholders, ",")
	err := m.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := m.db.PG.ExecContext(ctx, query, values...)
		return err
	})
	return err
}
