package vulnerabilitytotalstatistics

import (
	"context"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type vulnerabilityTotalStatisticsRepository struct {
	db *db.DB
}

func NewVulnerabilityTotalStatisticsRepository(db *db.DB) Repository {
	return &vulnerabilityTotalStatisticsRepository{db}
}

func (m *vulnerabilityTotalStatisticsRepository) DeleteByProjectID(ctx context.Context,
	projectID int) error {
	err := m.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := m.db.PG.Exec(
			"DELETE FROM VulnerabilityTotalStatistics "+
				" WHERE project_id = $1", projectID)
		return err
	})
	if err != nil {
		simplelog.Error("DeleteByProjectID ", "error", err.Error())
	}
	return err
}

func (m *vulnerabilityTotalStatisticsRepository) UpdateStatisticsByProjectID(ctx context.Context,
	projectID int) error {
	err := m.DeleteByProjectID(ctx, projectID)
	if err != nil {
		return err
	}

	err = m.db.Trier.Try(ctx, func(ctx context.Context) error {
		_, err := m.db.PG.Exec(
			"INSERT INTO VulnerabilityTotalStatistics "+
				" (organization_id, project_id, scan_type_id, scan_id, category_id,"+
				" total_vulnerabilities_count, total_not_false_count, total_not_reviewed_count, "+
				" blocker_not_false_count, blocker_not_reviewed_count, "+
				" critical_not_false_count, critical_not_reviewed_count, "+
				" medium_not_false_count, medium_not_reviewed_count, "+
				" low_not_false_count, low_not_reviewed_count, "+
				" info_not_false_count, info_not_reviewed_count) "+
				" ( "+
				" 	SELECT"+
				" 		o.id as organization_id, p.id as project_id, "+
				"		st.id as scan_type_id, s.id as scan_id, vc.id as category_id, "+
				"		COUNT(*) as total_vulnerabilities_count, "+
				"		COALESCE(SUM(CASE WHEN v.status IN ('not_reviewed', 'confirmed', 'to_verify') THEN 1 ELSE 0 END),0) as total_not_false_count, "+
				"		COALESCE(SUM(CASE WHEN v.status = 'not_reviewed' THEN 1 ELSE 0 END),0) as total_not_reviewed_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'blocker' AND v.status IN ('not_reviewed', 'confirmed', 'to_verify') THEN 1 ELSE 0 END),0) as blocker_not_false_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'blocker' AND v.status = 'not_reviewed' THEN 1 ELSE 0 END),0) as blocker_not_reviewed_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'critical' AND v.status IN ('not_reviewed', 'confirmed', 'to_verify') THEN 1 ELSE 0 END),0) as critical_not_false_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'critical' AND v.status = 'not_reviewed' THEN 1 ELSE 0 END),0) as critical_not_reviewed_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'medium' AND v.status IN ('not_reviewed', 'confirmed', 'to_verify') THEN 1 ELSE 0 END),0) as medium_not_false_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'medium' AND v.status = 'not_reviewed' THEN 1 ELSE 0 END),0) as medium_not_reviewed_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'low' AND v.status IN ('not_reviewed', 'confirmed', 'to_verify') THEN 1 ELSE 0 END),0) as low_not_false_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'low' AND v.status = 'not_reviewed' THEN 1 ELSE 0 END),0) as low_not_reviewed_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'info' AND v.status IN ('not_reviewed', 'confirmed', 'to_verify') THEN 1 ELSE 0 END),0) as info_not_false_count, "+
				"		COALESCE(SUM(CASE WHEN v.severity = 'info' AND v.status = 'not_reviewed' THEN 1 ELSE 0 END),0) as info_not_reviewed_count "+
				" 	FROM vulnerability as v "+
				"	JOIN vulnerability2scanInstance as v2si ON v2si.vulnerability_id = v.id "+
				"	JOIN vulnerabilitycategory as vc ON vc.id = v.category_id "+
				"	JOIN scan as s ON v.scan_id = s.id "+
				"	JOIN scantype as st ON s.scan_type_id = st.id "+
				"	JOIN project as p ON s.project_id = p.id "+
				"	JOIN organization as o ON p.organization_id = o.id "+
				"	WHERE"+
				"		p.id = $1 AND "+
				"		v2si.scan_instance_id IN ( "+
				"			SELECT MAX(si.id) as last_scan_instance_id "+
				"			FROM scanInstance si "+
				"			JOIN scan s ON si.scan_id = s.id "+
				"			JOIN project p ON p.id = s.project_id "+
				"			JOIN organization o ON o.id = p.organization_id "+
				"			GROUP BY o.id, p.id, si.scan_id) "+
				" 	GROUP BY o.id, p.id, st.id, s.id, vc.id "+
				" ) ", projectID)
		return err
	})
	if err != nil {
		simplelog.Error("UpdateStatisticsByProjectID ", "err", err.Error())
	}
	return err
}
