package cron

import (
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/drive/library/go/cron"
	"a.yandex-team.ru/security/impulse/api/internal/middlewares/access"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_cronRepo "a.yandex-team.ru/security/impulse/api/repositories/cron"
	"a.yandex-team.ru/security/impulse/api/scan-api/internal/infra"
	_cronUsecase "a.yandex-team.ru/security/impulse/api/usecases/cron"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	cronUsecase _cronUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	cronRepo := _cronRepo.NewCronRepository(c.DB)
	c.cronUsecase = _cronUsecase.NewCronUsecase(cronRepo)

	g.POST("/organizations/:organizationId/projects/:projectId/cron", c.new, access.WithRoleInProject(models.ADMIN))
	g.GET("/organizations/:organizationId/projects/:projectId/cron", c.list, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/cron/:id", c.getCron, access.WithRoleInProject(models.VIEW))
	g.PUT("/organizations/:organizationId/projects/:projectId/cron/:id", c.updateCron, access.WithRoleInProject(models.ADMIN))
	g.DELETE("/organizations/:organizationId/projects/:projectId/cron/:id", c.deleteCron, access.WithRoleInProject(models.ADMIN))
	g.PUT("/organizations/:organizationId/projects/:projectId/cron/:id/start", c.startCron, access.WithRoleInProject(models.ADMIN))
	g.PUT("/organizations/:organizationId/projects/:projectId/cron/:id/pause", c.pauseCron, access.WithRoleInProject(models.ADMIN))

	return nil
}

func (c *Controller) new(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	w := new(models.CronRequestDTO)
	if err := e.Bind(w); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	simplelog.Info(e.Request().Method+" "+e.Path(), "Spec", w.Spec)
	simplelog.Info(e.Request().Method+" "+e.Path(), "Description", w.Description)
	simplelog.Info(e.Request().Method+" "+e.Path(), "WorkflowId", w.WorkflowID)
	simplelog.Info(e.Request().Method+" "+e.Path(), "Parameters", w.Parameters)
	simplelog.Info(e.Request().Method+" "+e.Path(), "Analysers", w.Analysers)
	simplelog.Info(e.Request().Method+" "+e.Path(), "CallbackURL", w.CallbackURL)
	simplelog.Info(e.Request().Method+" "+e.Path(), "NonTemplateScan", w.NonTemplateScan)

	if organizationID == 0 || projectID == 0 || w.WorkflowID == "" && len(w.Analysers) == 0 || w.Spec == "" || w.Description == "" {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	spec, err := cron.Parse(w.Spec)
	if err != nil {
		return utils.APIError(e, err)
	}
	loc, _ := time.LoadLocation("Europe/Moscow")
	nextTime := spec.Next(time.Now().In(loc)).Unix()

	task := models.Cron{
		Spec:            w.Spec,
		NextTime:        nextTime,
		Description:     w.Description,
		IsRunning:       true,
		OrganizationID:  organizationID,
		ProjectID:       projectID,
		WorkflowID:      w.WorkflowID,
		Parameters:      w.Parameters,
		Analysers:       w.Analysers,
		CallbackURL:     w.CallbackURL,
		NonTemplateScan: w.NonTemplateScan,
	}
	cronID, err := c.cronUsecase.Create(e.Request().Context(), &task)
	if err != nil {
		return utils.APIError(e, err)
	}
	task.ID = cronID

	return utils.APIOk(e, task)
}

func (c *Controller) list(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(),
		"organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	tasks, err := c.cronUsecase.List(e.Request().Context(), organizationID, projectID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, tasks)
}

func (c *Controller) getCron(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(),
		"organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "id", e.Param("id"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	cronID, err := strconv.Atoi(e.Param("id"))
	if err != nil {
		return utils.APIError(e, err)
	}

	task, err := c.cronUsecase.GetByID(e.Request().Context(), organizationID, projectID, cronID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, task)
}

func (c *Controller) updateCron(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(),
		"organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "id", e.Param("id"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	cronID, err := strconv.Atoi(e.Param("id"))
	if err != nil {
		return utils.APIError(e, err)
	}
	w := new(models.CronUpdateRequestDTO)
	if err := e.Bind(w); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	task, err := c.cronUsecase.GetByID(e.Request().Context(), organizationID, projectID, cronID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if w.Spec != "" {
		spec, err := cron.Parse(w.Spec)
		if err != nil {
			return utils.APIError(e, err)
		}
		task.Spec = w.Spec
		loc, _ := time.LoadLocation("Europe/Moscow")
		task.NextTime = spec.Next(time.Now().In(loc)).Unix()
	}
	if w.Description != "" {
		task.Description = w.Description
	}
	if w.Parameters != nil {
		task.Parameters = w.Parameters
	}
	if len(w.Analysers) != 0 {
		task.Analysers = w.Analysers
	}
	task.CallbackURL = w.CallbackURL
	task.NonTemplateScan = w.NonTemplateScan

	err = c.cronUsecase.Update(e.Request().Context(), task)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, task)
}

func (c *Controller) deleteCron(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(),
		"organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "id", e.Param("id"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	cronID, err := strconv.Atoi(e.Param("id"))
	if err != nil {
		return utils.APIError(e, err)
	}

	err = c.cronUsecase.DeleteByID(e.Request().Context(), organizationID, projectID, cronID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) startCron(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(),
		"organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "id", e.Param("id"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	cronID, err := strconv.Atoi(e.Param("id"))
	if err != nil {
		return utils.APIError(e, err)
	}

	err = c.cronUsecase.UpdateRunningState(e.Request().Context(), organizationID, projectID, cronID, true)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) pauseCron(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(),
		"organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "id", e.Param("id"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	cronID, err := strconv.Atoi(e.Param("id"))
	if err != nil {
		return utils.APIError(e, err)
	}

	err = c.cronUsecase.UpdateRunningState(e.Request().Context(), organizationID, projectID, cronID, false)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}
