package idm

import (
	"encoding/json"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	_idmRepo "a.yandex-team.ru/security/impulse/api/repositories/idm"
	_orgRepo "a.yandex-team.ru/security/impulse/api/repositories/organization"
	_projectRepo "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepopository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	"a.yandex-team.ru/security/impulse/api/scan-api/internal/infra"
	_idmUsecase "a.yandex-team.ru/security/impulse/api/usecases/idm"
	_orgUsecase "a.yandex-team.ru/security/impulse/api/usecases/organization"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	orgUsecase     _orgUsecase.Usecase
	idmUsecase     _idmUsecase.Usecase
	projectUsecase _projectUsecase.Usecase
}

const (
	ORGANIZATION = "organization"
	PROJECT      = "project"
	ROLE         = "role"
)

func (c *Controller) BuildRoute(g *echo.Group) error {
	idmRepo := _idmRepo.NewIdmRepository(c.DB)
	c.idmUsecase = _idmUsecase.NewIdmUsecase(idmRepo)
	orgRepo := _orgRepo.NewOrganizationRepository(c.DB)
	projectRepo := _projectRepo.NewProjectRepository(c.DB)
	scanRepo := _scanRepopository.NewScanRepository(c.DB)
	scanInstanceRepo := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	scanInstanceUsecase := _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepo, scanInstanceRepo)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepo, scanInstanceRepo, scanInstanceUsecase)
	c.orgUsecase = _orgUsecase.NewOrganizationUsecase(orgRepo, c.projectUsecase, scanInstanceRepo, scanInstanceUsecase)

	g.GET("/info", c.info)
	g.POST("/add-role", c.addRole)
	g.POST("/remove-role", c.removeRole)
	g.GET("/get-all-roles", c.getAllRoles)
	return nil
}

func (c *Controller) info(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())

	organizations, err := c.orgUsecase.List(e.Request().Context())
	if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}

	roles := map[string]interface{}{
		models.VIEW:   models.NodeValue{Name: models.NodeName{Ru: "Доступ на просмотр", En: "View access"}},
		models.TRIAGE: models.NodeValue{Name: models.NodeName{Ru: "Управление сканированиями", En: "Triage access"}},
		models.ADMIN:  models.NodeValue{Name: models.NodeName{Ru: "Администратор", En: "Administrator"}},
	}
	roleNode := models.Node{
		Slug:   ROLE,
		Name:   models.NodeName{Ru: "Роль", En: "Role"},
		Values: roles,
	}

	organizationNode := models.Node{
		Slug:   ORGANIZATION,
		Name:   models.NodeName{Ru: "Организация", En: "Organization"},
		Values: make(map[string]interface{}),
	}

	for _, organization := range organizations {
		projectNode := models.Node{
			Slug:   PROJECT,
			Name:   models.NodeName{Ru: "Проект", En: "Project"},
			Values: make(map[string]interface{}),
		}
		projects, err := c.projectUsecase.ListByOrganizationID(e.Request().Context(), organization.ID)
		if err != nil {
			simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
			return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
		}
		for _, project := range projects {
			projectNode.Values[project.Slug] = models.NodeValue{
				Name:  models.NodeName{Ru: project.Name, En: project.Name},
				Roles: &roleNode,
			}
		}
		projectNode.Values["*"] = models.NodeValue{
			Name:  models.NodeName{Ru: "*", En: "*"},
			Roles: &roleNode,
		}

		organizationNode.Values[organization.Slug] = models.NodeValue{
			Name:  models.NodeName{Ru: organization.Name, En: organization.Name},
			Roles: &projectNode,
		}
	}
	organizationNode.Values["*"] = models.NodeValue{
		Name:  models.NodeName{Ru: "*", En: "*"},
		Roles: &roleNode,
	}

	return e.JSON(http.StatusOK, &models.IdmInfoResponseDTO{Code: 0, Roles: &organizationNode})
}

func (c *Controller) addRole(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	r := new(models.IdmAddRoleRequestDTO)
	if err := e.Bind(r); err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: err.Error()})
	}

	user, err := c.idmUsecase.GetOrCreateByLogin(e.Request().Context(), r.Login, r.SubjectType != "user")
	if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}
	simplelog.Info(e.Request().Method+" "+e.Path(), "user", user)

	var role models.IdmRoleDTO
	if err := json.Unmarshal([]byte(r.Role), &role); err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: err.Error()})
	}
	simplelog.Info(e.Request().Method+" "+e.Path(), "organization", role.Organization, "project", role.Project, "role", role.Role)

	if !c.idmUsecase.IsValidRole(role.Role) {
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: "Unknown role"})
	}

	idmUserRole := models.IdmUserRole{UserID: user.ID, Role: role.Role}
	if role.Organization == "*" {
		idmUserRole.OrganizationSlug = "*"
	} else {
		organizationID, err := c.orgUsecase.GetIDBySlug(e.Request().Context(), role.Organization)
		if err == db.ErrNotFound {
			return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: err.Error()})
		} else if err != nil {
			simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
			return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
		}

		idmUserRole.OrganizationSlug = role.Organization
		idmUserRole.OrganizationID = organizationID

		if role.Project == "*" {
			idmUserRole.ProjectSlug = "*"
		} else {
			projectID, err := c.projectUsecase.GetIDBySlug(e.Request().Context(), role.Project)
			if err == db.ErrNotFound {
				return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: err.Error()})
			} else if err != nil {
				simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
				return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
			}

			idmUserRole.ProjectSlug = role.Project
			idmUserRole.ProjectID = projectID
		}
	}

	err = c.idmUsecase.AddUserRole(e.Request().Context(), &idmUserRole)
	if err == _idmUsecase.ErrRoleAlreadyExists {
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Warning: "Already exists"})
	} else if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}

	return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 0})
}

func (c *Controller) removeRole(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	r := new(models.IdmRemoveRoleRequestDTO)
	if err := e.Bind(r); err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: err.Error()})
	}

	if r.Fired == 1 {
		err := c.idmUsecase.RemoveByLogin(e.Request().Context(), r.Login)
		if err != nil {
			simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
			return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
		}
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 0})
	}

	var role models.IdmRoleDTO
	if err := json.Unmarshal([]byte(r.Role), &role); err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Fatal: err.Error()})
	}
	simplelog.Info(e.Request().Method+" "+e.Path(), "organization", role.Organization, "project", role.Project, "role", role.Role)

	err := c.idmUsecase.RemoveUserRoleByLogin(e.Request().Context(), r.Login, &role)
	if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}

	userRoles, err := c.idmUsecase.ListUserRolesByLogin(e.Request().Context(), r.Login)
	if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}
	if len(userRoles) == 0 {
		err := c.idmUsecase.RemoveByLogin(e.Request().Context(), r.Login)
		if err != nil {
			simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
			return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
		}
	}
	return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 0})
}

func (c *Controller) getAllRoles(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	users, err := c.idmUsecase.ListUsers(e.Request().Context())
	if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}
	roles, err := c.idmUsecase.ListRoles(e.Request().Context())
	if err != nil {
		simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
		return e.JSON(http.StatusOK, &models.IdmResponseDTO{Code: 1, Error: err.Error()})
	}
	usersResponse := make([]*models.IdmUserAllRolesDTO, 0)
	for _, user := range users {
		subjectType := "user"
		if user.IsTvmApp {
			subjectType = "tvm_app"
		}
		userRoles := models.IdmUserAllRolesDTO{
			Login:       user.Login,
			Roles:       make([]*models.IdmRoleDTO, 0),
			SubjectType: subjectType,
		}
		for _, role := range roles {
			if role.UserID == user.ID {
				userRoles.Roles = append(userRoles.Roles, &models.IdmRoleDTO{
					Organization: role.OrganizationSlug,
					Project:      role.ProjectSlug,
					Role:         role.Role,
				})
			}
		}
		usersResponse = append(usersResponse, &userRoles)
	}

	return e.JSON(http.StatusOK, &models.IdmAllRolesResponseDTO{Code: 0, Users: usersResponse})
}
