package task

import (
	"encoding/json"
	"strconv"
	"time"

	"github.com/gofrs/uuid"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/callback"
	"a.yandex-team.ru/security/impulse/api/internal/middlewares/access"
	"a.yandex-team.ru/security/impulse/api/internal/user"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepopository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_taskRepo "a.yandex-team.ru/security/impulse/api/repositories/task"
	"a.yandex-team.ru/security/impulse/api/scan-api/internal/infra"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_taskUsecase "a.yandex-team.ru/security/impulse/api/usecases/task"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/impulse/pkg/queue"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra

	callbackAPI         callback.API
	projectUsecase      _projectUsecase.Usecase
	taskUsecase         _taskUsecase.Usecase
	scanInstanceUsecase _scanInstanceUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	c.callbackAPI = callback.New()
	c.callbackAPI.SetTVMClient(c.Infra.TVM)

	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	taskRepository := _taskRepo.NewTaskRepository(c.DB)
	c.taskUsecase = _taskUsecase.NewTaskUsecase(taskRepository)
	scanRepository := _scanRepopository.NewScanRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	c.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, c.scanInstanceUsecase)

	g.POST("/tasks", c.new)
	g.POST("/tasks_sandbox", c.newFromSandbox)
	g.GET("/tasks", c.list)
	g.GET("/tasks/:id/status", c.statusGet)
	g.POST("/tasks/:id/status", c.statusPost)

	g.POST("/organizations/:organizationId/tasks", c.newFromOrganization, access.WithRoleInOrganization(models.ADMIN))
	g.GET("/organizations/:organizationId/tasks", c.listFromOrganization, access.WithRoleInOrganization(models.VIEW))

	return nil
}

// swagger:operation POST /api/v1/control/tasks scan-api noReq
// ---
// summary: Start new task
// description: Start new task with given workflow id and repository with branch
// parameters:
// - name: body
//   in: body
//   description: Parameters to start workflow with
//   schema:
//     "$ref": "#/definitions/TaskRequestDTO"
//   required: true
// responses:
//   "200":
//     "$ref": "#/responses/apiResponse"
//   "500":
//     "$ref": "#/responses/apiResponse"
func (c *Controller) new(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	w := new(models.TaskRequestDTO)
	if err := e.Bind(w); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}
	simplelog.Info(e.Path(), "OrganizationID", w.OrganizationID)
	simplelog.Info(e.Path(), "ProjectID", w.ProjectID)
	simplelog.Info(e.Path(), "Parameters", w.Parameters)
	simplelog.Info(e.Path(), "Analysers", w.Analysers)
	simplelog.Info(e.Path(), "CallbackURL", w.CallbackURL)
	simplelog.Info(e.Path(), "NonTemplateScan", w.NonTemplateScan)

	if w.OrganizationID == 0 || w.ProjectID == 0 {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	if !user.HasAccessToProject(e, w.OrganizationID, w.ProjectID, models.ADMIN) {
		return utils.APIForbidden(e)
	}

	status := models.Created
	if len(w.Analysers) == 0 {
		status = models.Finished
	}
	taskID := uuid.Must(uuid.NewV4()).String()
	loc, _ := time.LoadLocation("Europe/Moscow")
	task := models.Task{
		TaskID:          taskID,
		OrganizationID:  w.OrganizationID,
		ProjectID:       w.ProjectID,
		Parameters:      w.Parameters,
		StartTime:       time.Now().In(loc).Unix(),
		Status:          status,
		CallbackURL:     w.CallbackURL,
		Analysers:       w.Analysers,
		NonTemplateScan: w.NonTemplateScan,
	}
	err := c.taskUsecase.Create(e.Request().Context(), &task)
	if err != nil {
		return utils.APIError(e, err)
	}

	if len(w.Analysers) == 0 {
		return utils.APIOk(e, echo.Map{
			"id": taskID,
		})
	}

	normalizedParameters, err := c.taskUsecase.NormalizeTaskParameters(w.Parameters)
	if err != nil {
		return utils.APIError(e, err)
	}

	msg, _ := json.Marshal(&models.TaskMessageDTO{
		OrganizationID: w.OrganizationID,
		ProjectID:      w.ProjectID,
		TaskID:         taskID,
		Parameters:     normalizedParameters,
		Analysers:      w.Analysers,
	})
	_, err = c.Queue.SendMessage(&queue.SendOptions{
		QueueURL: c.CFG.TasksQueueURL(),
		Msg:      string(msg),
	})
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{
		"id": taskID,
	})
}

func (c *Controller) newFromSandbox(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	w := new(models.TaskRequestFromSandboxDTO)
	if err := e.Bind(w); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}
	simplelog.Info(e.Path(), "OrganizationID", w.OrganizationID)
	simplelog.Info(e.Path(), "ProjectID", w.ProjectID)
	simplelog.Info(e.Path(), "Parameters", w.Parameters)
	simplelog.Info(e.Path(), "Analysers", w.Analysers)
	simplelog.Info(e.Path(), "SandboxTaskID", w.SandboxTaskID)

	if len(w.Analysers) == 0 || w.OrganizationID == 0 || w.ProjectID == 0 || w.SandboxTaskID == 0 {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	if !user.HasAccessToProject(e, w.OrganizationID, w.ProjectID, models.ADMIN) {
		return utils.APIForbidden(e)
	}

	taskID := uuid.Must(uuid.NewV4()).String()
	loc, _ := time.LoadLocation("Europe/Moscow")
	task := models.Task{
		TaskID:          taskID,
		OrganizationID:  w.OrganizationID,
		ProjectID:       w.ProjectID,
		Parameters:      w.Parameters,
		StartTime:       time.Now().In(loc).Unix(),
		Status:          models.Running,
		Analysers:       w.Analysers,
		SandboxTaskID:   w.SandboxTaskID,
		NonTemplateScan: w.NonTemplateScan,
	}
	err := c.taskUsecase.Create(e.Request().Context(), &task)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, echo.Map{
		"id": taskID,
	})
}

// swagger:operation GET /api/v1/control/tasks/{id}/status scan-api noReq
// ---
// summary: Get task status
// description: Receive status for task with given id
// parameters:
// - name: id
//   in: path
//   description: Task id
//   type: string
//   required: true
// responses:
//   "200":
//     "$ref": "#/responses/apiResponse"
//   "404":
//     "$ref": "#/responses/apiResponse"
func (c *Controller) statusGet(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "id", e.Param("id"))

	taskID := e.Param("id")
	task, err := c.taskUsecase.GetByTaskID(e.Request().Context(), taskID)
	if err != nil {
		return utils.APINotFound(e)
	}

	if !user.HasAccessToProject(e, task.OrganizationID, task.ProjectID, models.VIEW) {
		return utils.APIForbidden(e)
	}

	return utils.APIOk(e, echo.Map{
		"status": c.taskUsecase.StatusToString(task),
	})
}

// swagger:operation POST /api/v1/control/tasks/{id}/status scan-api noReq
// ---
// summary: Update task status
// description: Update status for task with given id
// parameters:
// - name: id
//   in: path
//   description: Task id
//   type: string
//   required: true
// - name: body
//   in: body
//   description: Status to set
//   schema:
//     "$ref": "#/definitions/TaskStatusDTO"
//   required: true
// responses:
//   "200":
//     "$ref": "#/responses/apiResponse"
//   "404":
//     "$ref": "#/responses/apiResponse"
//   "500":
//     "$ref": "#/responses/apiResponse"
func (c *Controller) statusPost(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "id", e.Param("id"))

	taskID := e.Param("id")
	w := new(models.TaskStatusDTO)
	if err := e.Bind(w); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}
	simplelog.Info(e.Path(), "status", w.Status)
	if !c.taskUsecase.IsValidStatus(w.Status) {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	task, err := c.taskUsecase.GetByTaskID(e.Request().Context(), taskID)
	if err != nil {
		return utils.APINotFound(e)
	}

	if !user.HasAccessToProject(e, task.OrganizationID, task.ProjectID, models.ADMIN) {
		return utils.APIForbidden(e)
	}

	callbackFailed := false
	if task.CallbackURL != "" {
		instances, err := c.scanInstanceUsecase.ListByTaskID(e.Request().Context(), taskID)
		if err != nil {
			simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
			callbackFailed = true
		} else {
			data, _ := json.Marshal(&models.WebhookReport{
				OrganizationID: task.OrganizationID,
				ProjectID:      task.ProjectID,
				Instances:      instances,
			})
			err = c.callbackAPI.Do(e.Request().Context(), task.CallbackURL, data)
			if err != nil {
				simplelog.Error(e.Request().Method+" "+e.Path(), "err", err)
				callbackFailed = true
			}
		}
	}

	task.Status = w.Status
	if task.Status == models.Finished && callbackFailed {
		task.Status = models.FinishedCallbackFailed
	}
	if c.taskUsecase.IsFinalStatus(w.Status) {
		loc, _ := time.LoadLocation("Europe/Moscow")
		task.EndTime = time.Now().In(loc).Unix()
	}
	err = c.taskUsecase.Update(e.Request().Context(), task)
	if err != nil {
		simplelog.Info(e.Path(), "err", err)
		return utils.APINotFound(e)
	}

	return utils.APIOk(e, echo.Map{})
}

// swagger:operation GET /api/v1/control/tasks scan-api noReq
// ---
// summary: List statuses of launched tasks
// description: List statuses of launched tasks
// parameters:
// - name: organization_id
//   in: query
//   description: Organization id
//   type: int
//   required: true
// - name: project_id
//   in: query
//   description: Project id
//   type: int
//   required: true
// - name: limit
//   in: query
//   description: Limit
//   type: int
//   required: false
// - name: offset
//   in: query
//   description: Offset
//   type: int
//   required: false
// responses:
//   "200":
//     "$ref": "#/responses/apiResponse"
//   "500":
//     "$ref": "#/responses/apiResponse"
func (c *Controller) list(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organization_id", e.QueryParam("organization_id"))
	simplelog.Info(e.Request().Method+" "+e.Path(), "project_id", e.QueryParam("project_id"))
	simplelog.Info(e.Request().Method+" "+e.Path(), "limit", e.QueryParam("limit"))
	simplelog.Info(e.Request().Method+" "+e.Path(), "offset", e.QueryParam("offset"))

	organizationID, _ := strconv.Atoi(e.QueryParam("organization_id"))
	projectID, _ := strconv.Atoi(e.QueryParam("project_id"))
	if organizationID <= 0 || projectID <= 0 {
		return utils.APIError(e, echo.ErrBadRequest)
	}
	if !user.HasAccessToProject(e, organizationID, projectID, models.VIEW) {
		return utils.APIForbidden(e)
	}
	limit, _ := strconv.Atoi(e.QueryParam("limit"))
	if limit <= 0 {
		limit = 50
	}
	offset, _ := strconv.Atoi(e.QueryParam("offset"))
	if offset < 0 {
		offset = 0
	}

	tasks, err := c.taskUsecase.List(e.Request().Context(), organizationID, projectID, limit, offset)
	if err != nil {
		return utils.APIError(e, err)
	}

	total, err := c.taskUsecase.GetTotal(e.Request().Context(), organizationID, projectID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, &models.TasksListResponseDTO{
		Total: total,
		Tasks: tasks,
	})
}

func (c *Controller) newFromOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	w := new(models.TaskOrganizationRequestDTO)
	if err := e.Bind(w); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}
	simplelog.Info(e.Path(), "Parameters", w.Parameters)
	simplelog.Info(e.Path(), "Analysers", w.Analysers)
	simplelog.Info(e.Path(), "Tags", w.Tags)

	if len(w.Analysers) == 0 || organizationID == 0 {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	msg, _ := json.Marshal(&models.TaskMessageDTO{
		OrganizationID: organizationID,
		Parameters:     w.Parameters,
		Analysers:      w.Analysers,
		OnAllProjects:  true,
		Tags:           w.Tags,
	})
	_, err = c.Queue.SendMessage(&queue.SendOptions{
		QueueURL: c.CFG.TasksQueueURL(),
		Msg:      string(msg),
	})
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) listFromOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	simplelog.Info(e.Request().Method+" "+e.Path(), "limit", e.QueryParam("limit"))
	simplelog.Info(e.Request().Method+" "+e.Path(), "offset", e.QueryParam("offset"))

	limit, _ := strconv.Atoi(e.QueryParam("limit"))
	if limit <= 0 {
		limit = 50
	}
	offset, _ := strconv.Atoi(e.QueryParam("offset"))
	if offset < 0 {
		offset = 0
	}

	tasks, err := c.taskUsecase.ListByOrganizationID(e.Request().Context(), organizationID, limit, offset)
	if err != nil {
		return utils.APIError(e, err)
	}

	total, err := c.taskUsecase.GetTotalByOrganizationID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, &models.TasksListResponseDTO{
		Total: total,
		Tasks: tasks,
	})
}
