package main

import (
	"context"
	"os"
	"os/signal"
	"strconv"
	"syscall"
	"time"

	"a.yandex-team.ru/security/impulse/api/scan-api/internal/config"
	"a.yandex-team.ru/security/impulse/api/scan-api/internal/infra"
	"a.yandex-team.ru/security/impulse/api/scan-api/internal/server"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	shutdownDeadline = time.Second * 30
)

func newConfig() config.Config {
	var cfg config.Config
	httpPort, err := strconv.Atoi(os.Getenv("HTTP_PORT"))
	if err != nil {
		httpPort = 80
	}
	switch os.Getenv("IMPULSE_ENV") {
	case "dev":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			Debug:         true,
			UseAuth:       false,
			UpstreamURL:   "http://localhost",
			DBConn:        os.Getenv("DB_DSN"),
			DBPassword:    os.Getenv("DB_PASSWORD"),
			DBRetries:     3,
			SqsEndpoint:   "https://sqs.yandex.net",
			SqsAccount:    "servicesecurity-dev",
			SqsRetries:    1,
			SqsOAuthToken: os.Getenv("SQS_OAUTH_TOKEN"),
			HTTPPort:      uint32(httpPort),
		}
	case "test":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			Debug:         true,
			UseAuth:       true,
			IdmTvmSrcID:   2001600,
			UpstreamURL:   "https://impulse.sec.yandex-team.ru",
			DBConn:        "host=c-mdbfts3u6vt7aop9seqf.rw.db.yandex.net port=6432 dbname=impulse sslmode=verify-full user=impulse",
			DBPassword:    os.Getenv("DB_PASSWORD"),
			DBRetries:     3,
			SqsEndpoint:   "https://sqs.yandex.net",
			SqsAccount:    "servicesecurity-test",
			SqsRetries:    5,
			SqsOAuthToken: os.Getenv("SQS_OAUTH_TOKEN"),
			HTTPPort:      uint32(httpPort),
		}
	case "prod":
		fallthrough
	default:
		cfg = config.Config{
			Debug:       false,
			UpstreamURL: "https://impulse.sec.yandex-team.ru",
			HTTPPort:    uint32(httpPort),
		}
	}

	return cfg
}

func main() {
	cfg := newConfig()
	srv, err := server.New(infra.New(cfg))
	if err != nil {
		panic(err)
	}

	// Setup quit channel
	quit := make(chan os.Signal)

	// Start server
	go func() {
		if err := srv.ListenAndServe(cfg.HTTPPort); err != nil {
			simplelog.Error("server server stopped", "err", err)
			// send signal manually, server failed to start
			signal.Stop(quit)
			quit <- syscall.SIGTERM
		}
	}()

	// Wait termination
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
	defer cancel()

	if err := srv.Shutdown(ctx); err != nil {
		simplelog.Error("failed to shutdown server", "err", err)
	}
}
