package organization

import (
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/middlewares/access"
	"a.yandex-team.ru/security/impulse/api/internal/user"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_organizationRepo "a.yandex-team.ru/security/impulse/api/repositories/organization"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	"a.yandex-team.ru/security/impulse/api/storage-api/internal/infra"
	_organizationUsecase "a.yandex-team.ru/security/impulse/api/usecases/organization"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	organizationUsecase _organizationUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	organizationRepo := _organizationRepo.NewOrganizationRepository(c.DB)
	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	scanInstanceUsecase := _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	projectUsecase := _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, scanInstanceUsecase)
	c.organizationUsecase = _organizationUsecase.NewOrganizationUsecase(organizationRepo,
		projectUsecase, scanInstanceRepository, scanInstanceUsecase)

	g.GET("/organizations", c.listOrganizations)
	g.POST("/organizations", c.createOrganization, access.WithRole(models.ADMIN))
	g.GET("/organizations/:organizationId", c.getOrganization, access.WithAnyRoleInOrganization)
	g.GET("/organizations/:organizationId/statistics", c.getOrganizationStatistics, access.WithAnyRoleInOrganization)
	g.PUT("/organizations/:organizationId", c.updateOrganization, access.WithRole(models.ADMIN))
	g.DELETE("/organizations/:organizationId", c.deleteOrganization, access.WithRole(models.ADMIN))
	return nil
}

func (c *Controller) createOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	orgRequestDTO := new(models.OrganizationRequestDTO)
	if err := e.Bind(orgRequestDTO); err != nil {
		return utils.APIError(e, err)
	}

	org, err := c.organizationUsecase.Create(e.Request().Context(), orgRequestDTO)
	if err != nil {
		return utils.APIError(e, err)
	}

	response := models.CreateOrganizationResponseDTO{ID: org.ID}
	return utils.APIOk(e, response)
}

func (c *Controller) updateOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))
	orgRequestDTO := new(models.OrganizationUpdateRequestDTO)
	if err := e.Bind(orgRequestDTO); err != nil {
		return utils.APIError(e, err)
	}
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	organization, err := c.organizationUsecase.GetByID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}

	if orgRequestDTO.Name != "" {
		organization.Name = orgRequestDTO.Name
	}
	organization.TrackerQueue = orgRequestDTO.TrackerQueue
	organization.ABCServiceID = orgRequestDTO.ABCServiceID
	err = c.organizationUsecase.Update(e.Request().Context(), organization)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, organization)
}

func (c *Controller) getOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	organization, err := c.organizationUsecase.GetByID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, organization)
}

func (c *Controller) getOrganizationStatistics(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	organization, err := c.organizationUsecase.GetByID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}
	organizationStatistcs, err := c.organizationUsecase.GetStatisticsByID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}
	organizationInfo := models.OrganizationInfo{
		Organization: *organization,
		Statistics:   *organizationStatistcs,
	}
	return utils.APIOk(e, organizationInfo)
}

func (c *Controller) listOrganizations(e echo.Context) error {
	simplelog.Info(e.Request().Method + " " + e.Path())
	organizations, err := c.organizationUsecase.List(e.Request().Context())
	if err != nil {
		return utils.APIError(e, err)
	}

	organizationsFiltered := make([]*models.OrganizationInfo, 0)
	for _, organization := range organizations {
		if user.HasAccessToAnyProject(e, organization.ID) {
			organizationsFiltered = append(organizationsFiltered, organization)
		}
	}

	return utils.APIOk(e, organizationsFiltered)
}

func (c *Controller) deleteOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))

	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	err = c.organizationUsecase.DeleteByID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}
