package project

import (
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/middlewares/access"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_scanTypeRepository "a.yandex-team.ru/security/impulse/api/repositories/scantype"
	_vulnerabilityRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerability"
	_vulnerabilityTotalStatisticsRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilitytotalstatistics"
	_workflowRepository "a.yandex-team.ru/security/impulse/api/repositories/workflow"
	"a.yandex-team.ru/security/impulse/api/storage-api/internal/infra"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanUsecase "a.yandex-team.ru/security/impulse/api/usecases/scan"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_scanTypeUsecase "a.yandex-team.ru/security/impulse/api/usecases/scantype"
	_vulnerabilityUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerability"
	_vulnerabilityTotalStatisticsUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitytotalstatistics"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	projectUsecase                      _projectUsecase.Usecase
	scanUsecase                         _scanUsecase.Usecase
	scanTypeUsecase                     _scanTypeUsecase.Usecase
	scanInstanceUsecase                 _scanInstanceUsecase.Usecase
	vulnerabilityUsecase                _vulnerabilityUsecase.Usecase
	vulnerabilityTotalStatisticsUsecase _vulnerabilityTotalStatisticsUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	scanTypeRepository := _scanTypeRepository.NewScanTypeRepository(c.DB)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	c.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, c.scanInstanceUsecase)
	vulnerabilityRepo := _vulnerabilityRepo.NewVulnerabilityRepository(c.DB)
	vulnerabilityTotalStatisticsRepo := _vulnerabilityTotalStatisticsRepo.NewVulnerabilityTotalStatisticsRepository(c.DB)
	c.vulnerabilityTotalStatisticsUsecase = _vulnerabilityTotalStatisticsUsecase.NewVulnerabilityTotalStatisticsUsecase(vulnerabilityTotalStatisticsRepo)
	c.vulnerabilityUsecase = _vulnerabilityUsecase.NewVulnerabilityUsecase(vulnerabilityRepo, c.vulnerabilityTotalStatisticsUsecase)
	c.scanTypeUsecase = _scanTypeUsecase.NewScanTypeUsecase(scanTypeRepository)
	workflowRepository := _workflowRepository.NewWorkflowRepository(c.DB)
	c.scanUsecase = _scanUsecase.NewScanUsecase(scanRepository, projectRepository, scanTypeRepository,
		scanInstanceRepository, workflowRepository)

	g.GET("/organizations/:organizationId/projects", c.listProjects, access.WithAnyRoleInOrganization)
	g.GET("/organizations/:organizationId/projects/:projectId", c.getProject, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/statistics", c.getProjectStatistics, access.WithRoleInProject(models.VIEW))
	g.POST("/organizations/:organizationId/projects", c.createProject, access.WithRoleInOrganization(models.ADMIN))
	g.PUT("/organizations/:organizationId/projects/:projectId", c.updateProject, access.WithRoleInProject(models.ADMIN))
	g.DELETE("/organizations/:organizationId/projects/:projectId", c.deleteProject, access.WithRoleInProject(models.ADMIN))
	g.POST("/organizations/:organizationId/projects/:projectId/createScansFromAnalysersList", c.createScansFromAnalysersList, access.WithRoleInProject(models.ADMIN))
	g.GET("/organizations/:organizationId/projects/:projectId/scans", c.listProjectScanTypes, access.WithRoleInProject(models.VIEW))
	g.GET("/tags", c.listTags, access.WithRole(models.VIEW))

	return nil
}

func (c *Controller) listProjects(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))

	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	projects, err := c.projectUsecase.ListByOrganizationID(e.Request().Context(), organizationID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, projects)
}

func (c *Controller) createProject(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"))

	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	projectFromRequest := new(models.Project)
	if err := e.Bind(projectFromRequest); err != nil {
		return utils.APIError(e, err)
	}
	projectFromRequest.OrganizationID = organizationID

	createdProject, err := c.projectUsecase.Create(e.Request().Context(), projectFromRequest)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, createdProject)
}

func (c *Controller) getProject(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	return utils.APIOk(e, project)
}

func (c *Controller) getProjectStatistics(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	projectStatistics, err := c.projectUsecase.GetStatisticsByID(e.Request().Context(), project.ID)
	if err != nil {
		return utils.APIError(e, err)
	}
	projectInfo := models.ProjectInfo{Project: *project, Statistics: *projectStatistics}

	return utils.APIOk(e, projectInfo)
}

func (c *Controller) updateProject(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	projectFromRequest := new(models.ProjectUpdateRequestDTO)
	if err := e.Bind(projectFromRequest); err != nil {
		return utils.APIError(e, err)
	}

	if projectFromRequest.Name != "" {
		project.Name = projectFromRequest.Name
	}

	if projectFromRequest.OrganizationID != 0 {
		project.OrganizationID = projectFromRequest.OrganizationID
	}

	project.TrackerQueue = projectFromRequest.TrackerQueue
	project.ABCServiceID = projectFromRequest.ABCServiceID
	project.Tags = projectFromRequest.Tags
	project.BuildCodeQLIndex = projectFromRequest.BuildCodeQLIndex
	project.NotificationSettings = projectFromRequest.NotificationSettings

	err = c.projectUsecase.Update(e.Request().Context(), project)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, project)
}

func (c *Controller) deleteProject(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	err = c.projectUsecase.DeleteByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) createScansFromAnalysersList(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))

	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	var analysersFromRequest models.TaskAnalysersRequestDTO
	if err := e.Bind(&analysersFromRequest); err != nil {
		return utils.APIError(e, err)
	}

	_, err = c.scanUsecase.CreateScansFromAnalysersList(e.Request().Context(), projectID, analysersFromRequest.Analysers)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) listProjectScanTypes(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	scanTypes, err := c.scanTypeUsecase.ListByProjectID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, scanTypes)
}

func (c *Controller) listTags(e echo.Context) error {
	simplelog.Info(e.Request().Method)

	tags, err := c.projectUsecase.ListTags(e.Request().Context())
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, tags)
}
