package scan

import (
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/middlewares/access"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_scanTypeRepository "a.yandex-team.ru/security/impulse/api/repositories/scantype"
	_workflowRepository "a.yandex-team.ru/security/impulse/api/repositories/workflow"
	"a.yandex-team.ru/security/impulse/api/storage-api/internal/infra"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanUsecase "a.yandex-team.ru/security/impulse/api/usecases/scan"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_scanTypeUsecase "a.yandex-team.ru/security/impulse/api/usecases/scantype"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	projectUsecase      _projectUsecase.Usecase
	scanUsecase         _scanUsecase.Usecase
	scanTypeUsecase     _scanTypeUsecase.Usecase
	scanInstanceUsecase _scanInstanceUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanTypeRepository := _scanTypeRepository.NewScanTypeRepository(c.DB)
	workflowRepository := _workflowRepository.NewWorkflowRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	c.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, c.scanInstanceUsecase)
	c.scanUsecase = _scanUsecase.NewScanUsecase(scanRepository, projectRepository, scanTypeRepository,
		scanInstanceRepository, workflowRepository)
	c.scanTypeUsecase = _scanTypeUsecase.NewScanTypeUsecase(scanTypeRepository)

	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName", c.getScanInfo,
		access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances", c.getScanInstances,
		access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId",
		c.getScanInstanceInfo, access.WithRoleInProject(models.VIEW))
	g.DELETE("/organizations/:organizationId/projects/:projectId/scans/:scanId", c.deleteScan,
		access.WithRoleInProject(models.ADMIN))

	return nil
}

func (c *Controller) deleteScan(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanId", e.Param("scanId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	scanID, err := strconv.Atoi(e.Param("scanId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scan, err := c.scanUsecase.GetByID(e.Request().Context(), scanID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.ID != scan.ProjectID {
		return utils.APINotFound(e)
	}

	err = c.scanUsecase.DeleteByID(e.Request().Context(), scanID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) getScanInfo(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanTypeName := e.Param("scanTypeName")
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	scanInfo, err := c.scanUsecase.GetInfoByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, scanInfo)
}

func (c *Controller) getScanInstances(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByOrganizationIDAndProjectID(e.Request().Context(), organizationID, projectID)
	if err != nil {
		return utils.APIError(e, err)
	}

	scanTypeName := e.Param("scanTypeName")
	scan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), project.ID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}

	scanInstances, err := c.scanInstanceUsecase.ListByScanID(e.Request().Context(), scan.ID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, scanInstances)
}

func (c *Controller) getScanInstanceInfo(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"), "scanInstanceId",
		e.Param("scanInstanceId"))

	// TODO: check organizationId, projectId, scanTypeName, scanInstanceId
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstanceInfo, err := c.scanInstanceUsecase.GetInfoByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, scanInstanceInfo)
}
