package task

import (
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/user"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_taskRepo "a.yandex-team.ru/security/impulse/api/repositories/task"
	"a.yandex-team.ru/security/impulse/api/storage-api/internal/infra"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_taskUsecase "a.yandex-team.ru/security/impulse/api/usecases/task"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	taskUsecase         _taskUsecase.Usecase
	scanInstanceUsecase _scanInstanceUsecase.Usecase
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	taskRepo := _taskRepo.NewTaskRepository(c.DB)
	c.taskUsecase = _taskUsecase.NewTaskUsecase(taskRepo)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	c.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)

	g.GET("/tasks/:taskId/scanInstances", c.getScanInstances)
	g.GET("/tasks/:taskId/statistics", c.getTaskStatistics)

	return nil
}

func (c *Controller) getTaskStatistics(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "taskId", e.Param("taskId"))

	taskID := e.Param("taskId")
	task, err := c.taskUsecase.GetByTaskID(e.Request().Context(), taskID)
	if err != nil {
		return utils.APINotFound(e)
	}
	if !user.HasAccessToProject(e, task.OrganizationID, task.ProjectID, models.VIEW) {
		return utils.APIForbidden(e)
	}

	scanInstances, err := c.scanInstanceUsecase.ListByTaskID(e.Request().Context(), taskID)
	if err != nil {
		return utils.APIError(e, err)
	}

	taskStatistics, err := c.scanInstanceUsecase.GetSummaryStatisticsFromScanInstances(e.Request().Context(), scanInstances)
	if err != nil {
		return utils.APIError(e, err)
	}

	taskInfo := models.TaskInfo{Task: *task, ScanInstanceStatistics: *taskStatistics}

	return utils.APIOk(e, taskInfo)
}

func (c *Controller) getScanInstances(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "taskId", e.Param("taskId"))

	taskID := e.Param("taskId")
	task, err := c.taskUsecase.GetByTaskID(e.Request().Context(), taskID)
	if err != nil {
		return utils.APINotFound(e)
	}
	if !user.HasAccessToProject(e, task.OrganizationID, task.ProjectID, models.VIEW) {
		return utils.APIForbidden(e)
	}

	scanInstances, err := c.scanInstanceUsecase.ListByTaskID(e.Request().Context(), taskID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, scanInstances)
}
