package vuln

import (
	"context"
	"encoding/json"
	"strconv"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/middlewares/access"
	"a.yandex-team.ru/security/impulse/api/internal/secnotify"
	"a.yandex-team.ru/security/impulse/api/internal/user"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_projectRepo "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepo "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepo "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_scanTypeRepo "a.yandex-team.ru/security/impulse/api/repositories/scantype"
	_taskRepo "a.yandex-team.ru/security/impulse/api/repositories/task"
	_vulnerabilityRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerability"
	_vulnerabilityCategoryRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilitycategory"
	_vulnerabilityHistoryActionRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilityhistoryaction"
	_vulnerabilityTotalStatisticsRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilitytotalstatistics"
	_workflowRepo "a.yandex-team.ru/security/impulse/api/repositories/workflow"
	"a.yandex-team.ru/security/impulse/api/storage-api/internal/infra"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanUsecase "a.yandex-team.ru/security/impulse/api/usecases/scan"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_taskUsecase "a.yandex-team.ru/security/impulse/api/usecases/task"
	_vulnerabilityUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerability"
	_vulnerabilityCategoryUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitycategory"
	_vulnerabilityHistoryActionUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilityhistoryaction"
	_vulnerabilityTotalStatisticsUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitytotalstatistics"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type Controller struct {
	*infra.Infra
	scanUsecase                          _scanUsecase.Usecase
	vulnerabilityUsecase                 _vulnerabilityUsecase.Usecase
	vulnerabilityCategoryUsecase         _vulnerabilityCategoryUsecase.Usecase
	vulnerabilityHistoryActionUsecase    _vulnerabilityHistoryActionUsecase.Usecase
	vulnerabilityTotalStatisticsUsecase  _vulnerabilityTotalStatisticsUsecase.Usecase
	scanInstanceUsecase                  _scanInstanceUsecase.Usecase
	projectUsecase                       _projectUsecase.Usecase
	taskUsecase                          _taskUsecase.Usecase
	updateVulnerabilityStatProjectIDChan chan int
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	scanRepo := _scanRepo.NewScanRepository(c.DB)
	projectRepo := _projectRepo.NewProjectRepository(c.DB)
	scanTypeRepo := _scanTypeRepo.NewScanTypeRepository(c.DB)
	workflowRepo := _workflowRepo.NewWorkflowRepository(c.DB)
	scanInstanceRepo := _scanInstanceRepo.NewScanInstanceRepository(c.DB)
	taskRepo := _taskRepo.NewTaskRepository(c.DB)
	c.scanUsecase = _scanUsecase.NewScanUsecase(scanRepo, projectRepo, scanTypeRepo, scanInstanceRepo, workflowRepo)
	c.taskUsecase = _taskUsecase.NewTaskUsecase(taskRepo)
	vulnerabilityRepo := _vulnerabilityRepo.NewVulnerabilityRepository(c.DB)
	vulnerabilityTotalStatisticsRepo := _vulnerabilityTotalStatisticsRepo.NewVulnerabilityTotalStatisticsRepository(c.DB)
	c.vulnerabilityTotalStatisticsUsecase = _vulnerabilityTotalStatisticsUsecase.NewVulnerabilityTotalStatisticsUsecase(vulnerabilityTotalStatisticsRepo)
	c.vulnerabilityUsecase = _vulnerabilityUsecase.NewVulnerabilityUsecase(vulnerabilityRepo, c.vulnerabilityTotalStatisticsUsecase)
	vulnerabilityCategoryRepo := _vulnerabilityCategoryRepo.NewVulnerabilityCategoryRepository(c.DB)
	c.vulnerabilityCategoryUsecase = _vulnerabilityCategoryUsecase.NewVulnerabilityCategoryUsecase(vulnerabilityCategoryRepo)
	c.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepo, scanInstanceRepo)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepo, scanInstanceRepo, c.scanInstanceUsecase)
	vulnerabilityHistoryActionRepo := _vulnerabilityHistoryActionRepo.NewVulnerabilityHistoryActionRepository(c.DB)
	c.vulnerabilityHistoryActionUsecase = _vulnerabilityHistoryActionUsecase.NewVulnerabilityHistoryActionUsecase(vulnerabilityHistoryActionRepo)

	g.GET("/organizations/:organizationId/projects/:projectId/vulnerabilities/:scanTypeName", c.getVulnerabilities, access.WithRoleInProject(models.VIEW))
	g.POST("/organizations/:organizationId/projects/:projectId/vulnerabilities/:scanTypeName", c.storeVulnerabilities, access.WithRoleInProject(models.ADMIN))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities",
		c.getVulnerabilitiesFromScanInstance, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities/:vulnerabilityId",
		c.getVulnerability, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities/:vulnerabilityId/history",
		c.getVulnerabilityHistory, access.WithRoleInProject(models.VIEW))
	g.POST("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities/:vulnerabilityId/add_comment",
		c.addVulnerabilityComment, access.WithRoleInProject(models.TRIAGE))
	g.POST("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities/:vulnerabilityId/status",
		c.updateVulnerabilityStatus, access.WithRoleInProject(models.TRIAGE))
	g.POST("/organizations/:organizationId/projects/:projectId/vulnerabilities/batch_status_update",
		c.batchUpdateVulnerabilityStatus, access.WithRoleInProject(models.TRIAGE))
	g.POST("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities/:vulnerabilityId/tracker_ticket",
		c.updateVulnerabilityTrackerTicket, access.WithRoleInProject(models.TRIAGE))
	g.POST("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/vulnerabilities/:vulnerabilityId/create_tracker_ticket",
		c.createVulnerabilityTrackerTicket, access.WithRoleInProject(models.TRIAGE))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/categories",
		c.getCategories, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/categories",
		c.getCategoriesForInstance, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/categories/:categoryId/vulnerabilities",
		c.getScanInstanceCategoryVulnerabilities, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/scans/:scanTypeName/instances/:scanInstanceId/categories/:categoryId",
		c.getScanInstanceCategoryInfo, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/projects/:projectId/vulnerabilities",
		c.getVulnerabilitiesFromProject, access.WithRoleInProject(models.VIEW))
	g.GET("/organizations/:organizationId/vulnerabilities",
		c.getVulnerabilitiesFromOrganization, access.WithRoleInOrganization(models.VIEW))

	c.updateVulnerabilityStatProjectIDChan = make(chan int)
	go c.updateVulnerabilityStatWorker(c.updateVulnerabilityStatProjectIDChan)
	return nil
}

func (c *Controller) updateVulnerabilityStatWorker(projectIDs <-chan int) {
	for projectID := range projectIDs {
		_ = c.vulnerabilityTotalStatisticsUsecase.UpdateStatisticsByProjectID(context.TODO(),
			projectID)
	}
}

func (c *Controller) getVulnerabilities(e echo.Context) error {
	simplelog.Info(e.Path(), "organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"))

	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanTypeName := e.Param("scanTypeName")

	currentScan, err := c.scanUsecase.GetOrCreateByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}

	vulnerabilities, err := c.vulnerabilityUsecase.FetchByScanID(e.Request().Context(), currentScan.ID)
	if err != nil {
		return utils.APIError(e, err)
	}

	response := new(models.ProjectVulnerabilitiesDeduplicationResponseDTO)
	response.ID = currentScan.ProjectID
	response.LastUpdateToken = currentScan.LastUpdateToken
	response.Vulnerabilities = make([]*models.VulnerabilityDeduplicationResponseDTO, 0)
	for _, vulnerability := range vulnerabilities {
		response.Vulnerabilities = append(response.Vulnerabilities,
			&models.VulnerabilityDeduplicationResponseDTO{ID: vulnerability.ID, KeyProperties: vulnerability.KeyProperties})
	}

	return utils.APIOk(e, response)
}

func (c *Controller) storeVulnerabilities(e echo.Context) error {
	simplelog.Info(e.Path(), "organizationId", e.Param("organizationId"), "projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"))

	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanTypeName := e.Param("scanTypeName")

	currentScan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}

	deduplicationRequest := new(models.ProjectVulnerabilitiesDeduplicationRequestDTO)
	err = e.Bind(deduplicationRequest)
	if err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	categoryMap, err := c.vulnerabilityCategoryUsecase.ListByScanType(e.Request().Context(), currentScan.ScanTypeID)
	if err != nil {
		return utils.APIError(e, err)
	}

	insertedVulnerabilities := make([]*models.Vulnerability, 0)
	if len(deduplicationRequest.NewVulnerabilities) > 0 {
		newVulnerabilities := make([]*models.Vulnerability, 0)
		for _, vulnerabilityDTO := range deduplicationRequest.NewVulnerabilities {
			categoryID, ok := categoryMap[vulnerabilityDTO.Category]
			if !ok {
				categoryID, err = c.vulnerabilityCategoryUsecase.Insert(e.Request().Context(), currentScan.ScanTypeID, vulnerabilityDTO.Category)
				if err != nil {
					return utils.APIError(e, err)
				}
				categoryMap[vulnerabilityDTO.Category] = categoryID
			}
			newVulnerabilities = append(newVulnerabilities,
				&models.Vulnerability{
					ScanID:            currentScan.ID,
					Severity:          &vulnerabilityDTO.Severity,
					CategoryID:        categoryID,
					KeyProperties:     vulnerabilityDTO.KeyProperties,
					DisplayProperties: vulnerabilityDTO.DisplayProperties,
				})
		}

		insertedVulnerabilities, err = c.vulnerabilityUsecase.InsertByScanIDAndLastUpdateToken(e.Request().Context(),
			currentScan.ID, deduplicationRequest.LastUpdateToken, newVulnerabilities)
		if err != nil {
			return utils.APIError(e, err)
		}
	}

	deduplecatedVulnerabilities := make([]*models.Vulnerability, 0)
	for _, vulnerabilityDTO := range deduplicationRequest.DeduplicatedVulnerabilities {
		categoryID, ok := categoryMap[vulnerabilityDTO.Category]
		if !ok {
			categoryID, err = c.vulnerabilityCategoryUsecase.Insert(e.Request().Context(), currentScan.ScanTypeID, vulnerabilityDTO.Category)
			if err != nil {
				return utils.APIError(e, err)
			}
			categoryMap[vulnerabilityDTO.Category] = categoryID
		}
		deduplecatedVulnerabilities = append(deduplecatedVulnerabilities,
			&models.Vulnerability{
				ID:                vulnerabilityDTO.ID,
				ScanID:            currentScan.ID,
				Severity:          &vulnerabilityDTO.Severity,
				CategoryID:        categoryID,
				KeyProperties:     vulnerabilityDTO.KeyProperties,
				DisplayProperties: vulnerabilityDTO.DisplayProperties,
			})
	}
	currentScanInstance, err := c.scanInstanceUsecase.Create(e.Request().Context(), &models.ScanInstance{
		ScanID:       currentScan.ID,
		TaskID:       deduplicationRequest.TaskID,
		RawReportURL: deduplicationRequest.RawReportURL,
		ReportURL:    deduplicationRequest.ReportURL,
		CommitHash:   deduplicationRequest.CommitHash,
		StartTime:    time.Unix(deduplicationRequest.StartTime, 0),
		EndTime:      time.Unix(deduplicationRequest.EndTime, 0),
	})
	if err != nil {
		return utils.APIError(e, err)
	}

	allVulnerabilities := make([]*models.Vulnerability, 0)
	allVulnerabilities = append(allVulnerabilities, insertedVulnerabilities...)
	allVulnerabilities = append(allVulnerabilities, deduplecatedVulnerabilities...)

	if len(allVulnerabilities) > 0 {
		_, err = c.scanInstanceUsecase.InsertVulnerabilities2ScanInstance(e.Request().Context(), currentScanInstance,
			allVulnerabilities)
		if err != nil {
			return utils.APIError(e, err)
		}

		err = c.vulnerabilityUsecase.UpdateFromScanInstanceVulnerabilities(e.Request().Context(), currentScanInstance.ID)
		if err != nil {
			return utils.APIError(e, err)
		}
	}

	c.updateVulnerabilityStatProjectIDChan <- projectID

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) getVulnerabilitiesFromScanInstance(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"), "scanInstanceId", e.Param("scanInstanceId"),
		"limit", e.QueryParam("limit"), "offset", e.QueryParam("offset"))
	limit, err := strconv.Atoi(e.QueryParam("limit"))
	if err != nil {
		limit = 50
	}
	offset, err := strconv.Atoi(e.QueryParam("offset"))
	if err != nil {
		offset = 0
	}

	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APINotFound(e)
	}
	scan, err := c.scanUsecase.GetByID(e.Request().Context(), scanInstance.ScanID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if scan.ProjectID != project.ID {
		return utils.APINotFound(e)
	}

	vulnerabilities, err := c.vulnerabilityUsecase.FetchByScanInstanceID(e.Request().Context(), scanInstanceID,
		limit, offset)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, vulnerabilities)
}

func (c *Controller) getVulnerabilitiesFromProject(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "limit", e.QueryParam("limit"),
		"offset", e.QueryParam("offset"), "filter", e.QueryParam("filter"), "taskID", e.QueryParam("taskID"))
	limit, err := strconv.Atoi(e.QueryParam("limit"))
	if err != nil {
		limit = 50
	}
	offset, err := strconv.Atoi(e.QueryParam("offset"))
	if err != nil {
		offset = 0
	}
	filterStr := e.QueryParam("filter")
	if filterStr == `` {
		filterStr = `{}`
	}
	var filterMap map[string]interface{}
	err = json.Unmarshal([]byte(filterStr), &filterMap)
	if err != nil {
		return utils.APIError(e, err)
	}
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	taskID := e.QueryParam("taskId")
	var scanInstances []*models.ScanInstance
	if taskID == "" || taskID == "null" {
		simplelog.Info("Generating vulns for last scan")
		scanInstances, err = c.scanInstanceUsecase.ListLastScanInstancesByProjectID(e.Request().Context(), projectID)
	} else {
		simplelog.Info("Generating vulns for", "taskId", e.QueryParam("taskId"))
		task, err := c.taskUsecase.GetByTaskID(e.Request().Context(), taskID)
		if err != nil {
			return utils.APINotFound(e)
		}
		if !user.HasAccessToProject(e, task.OrganizationID, task.ProjectID, models.VIEW) {
			return utils.APIForbidden(e)
		}
		scanInstances, err = c.scanInstanceUsecase.ListByTaskID(e.Request().Context(), taskID)
		if err != nil {
			return utils.APIError(e, err)
		}
	}
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilities, err := c.vulnerabilityUsecase.FetchByScanInstances(e.Request().Context(), scanInstances,
		limit, offset, filterMap)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, vulnerabilities)
}

func (c *Controller) getVulnerabilitiesFromOrganization(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"limit", e.QueryParam("limit"), "offset", e.QueryParam("offset"),
		"filter", e.QueryParam("filter"))
	limit, err := strconv.Atoi(e.QueryParam("limit"))
	if err != nil {
		limit = 50
	}
	offset, err := strconv.Atoi(e.QueryParam("offset"))
	if err != nil {
		offset = 0
	}
	filterStr := e.QueryParam("filter")
	if filterStr == `` {
		filterStr = `{}`
	}
	var filterMap map[string]interface{}
	err = json.Unmarshal([]byte(filterStr), &filterMap)
	if err != nil {
		return utils.APIError(e, err)
	}
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	vulnerabilities, err := c.vulnerabilityUsecase.FetchLatestByOrganizationID(e.Request().Context(),
		organizationID, limit, offset, filterMap)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, vulnerabilities)
}

func (c *Controller) getVulnerability(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "vulnerabilityId", e.Param("vulnerabilityId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	scanTypeName := e.Param("scanTypeName")
	scan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if scanInstance.ScanID != scan.ID {
		return utils.APINotFound(e)
	}
	vulnerabilityID, err := strconv.Atoi(e.Param("vulnerabilityId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	vulnerability, err := c.vulnerabilityUsecase.GetByVulnerabilityIDAndScanInstanceID(e.Request().Context(),
		vulnerabilityID, scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, vulnerability)
}

func (c *Controller) getVulnerabilityHistory(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "vulnerabilityId", e.Param("vulnerabilityId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	scanTypeName := e.Param("scanTypeName")
	scan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if scanInstance.ScanID != scan.ID {
		return utils.APINotFound(e)
	}
	vulnerabilityID, err := strconv.Atoi(e.Param("vulnerabilityId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	vulnerabilityHistoryActions, err := c.vulnerabilityHistoryActionUsecase.FetchByVulnerabilityID(e.Request().Context(),
		vulnerabilityID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, vulnerabilityHistoryActions)
}

func (c *Controller) addVulnerabilityComment(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "vulnerabilityId", e.Param("vulnerabilityId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	scanTypeName := e.Param("scanTypeName")
	scan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if scanInstance.ScanID != scan.ID {
		return utils.APINotFound(e)
	}
	vulnerabilityID, err := strconv.Atoi(e.Param("vulnerabilityId"))
	if err != nil {
		return utils.APIError(e, err)
	}

	vulnerabilityCommentRequestDTO := new(models.VulnerabilityCommentRequestDTO)
	if err := e.Bind(vulnerabilityCommentRequestDTO); err != nil {
		return utils.APIError(e, err)
	}

	currentUser := e.Get("user").(*user.User)

	err = c.vulnerabilityHistoryActionUsecase.AddComment(e.Request().Context(), vulnerabilityID,
		currentUser.Login, vulnerabilityCommentRequestDTO.CommentText)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) updateVulnerabilityStatus(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "vulnerabilityId", e.Param("vulnerabilityId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	scanTypeName := e.Param("scanTypeName")
	scan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if scanInstance.ScanID != scan.ID {
		return utils.APINotFound(e)
	}
	vulnerabilityID, err := strconv.Atoi(e.Param("vulnerabilityId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilityUpdateStatusRequestDTO := new(models.VulnerabilityUpdateStatusRequestDTO)
	if err := e.Bind(vulnerabilityUpdateStatusRequestDTO); err != nil {
		return utils.APIError(e, err)
	}
	err = c.vulnerabilityUsecase.UpdateStatusByID(e.Request().Context(), vulnerabilityUpdateStatusRequestDTO.Status,
		vulnerabilityID)
	if err != nil {
		return utils.APIError(e, err)
	}
	// Add comment to History
	currentUser := e.Get("user").(*user.User)
	err = c.vulnerabilityHistoryActionUsecase.AddStatusChange(e.Request().Context(), vulnerabilityID,
		currentUser.Login, vulnerabilityUpdateStatusRequestDTO.Status)
	if err != nil {
		return utils.APIError(e, err)
	}

	c.updateVulnerabilityStatProjectIDChan <- projectID

	return utils.APIOk(e, echo.Map{})
}
func (c *Controller) batchUpdateVulnerabilityStatus(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}

	batchUpdateVulnerabilityStatusRequestDTO := new(models.BatchUpdateVulnerabilityStatusRequestDTO)
	if err := e.Bind(batchUpdateVulnerabilityStatusRequestDTO); err != nil {
		return utils.APIError(e, err)
	}
	err = c.vulnerabilityUsecase.BatchUpdateStatus(e.Request().Context(), batchUpdateVulnerabilityStatusRequestDTO.Status,
		batchUpdateVulnerabilityStatusRequestDTO.Vulnerabilities)
	if err != nil {
		return utils.APIError(e, err)
	}
	// Add comment to History
	currentUser := e.Get("user").(*user.User)
	err = c.vulnerabilityHistoryActionUsecase.BatchAddStatusChange(e.Request().Context(),
		batchUpdateVulnerabilityStatusRequestDTO.Vulnerabilities, currentUser.Login,
		batchUpdateVulnerabilityStatusRequestDTO.Status)
	if err != nil {
		return utils.APIError(e, err)
	}

	c.updateVulnerabilityStatProjectIDChan <- projectID

	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) updateVulnerabilityTrackerTicket(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "vulnerabilityId", e.Param("vulnerabilityId"))
	organizationID, err := strconv.Atoi(e.Param("organizationId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	project, err := c.projectUsecase.GetByID(e.Request().Context(), projectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if project.OrganizationID != organizationID {
		return utils.APINotFound(e)
	}
	scanTypeName := e.Param("scanTypeName")
	scan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}
	if scanInstance.ScanID != scan.ID {
		return utils.APINotFound(e)
	}
	vulnerabilityID, err := strconv.Atoi(e.Param("vulnerabilityId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilityUpdateTrackerTicketRequestDTO := new(models.VulnerabilityUpdateTrackerTicketRequestDTO)
	if err := e.Bind(vulnerabilityUpdateTrackerTicketRequestDTO); err != nil {
		return utils.APIError(e, err)
	}
	err = c.vulnerabilityUsecase.UpdateTrackerTicketByID(e.Request().Context(),
		vulnerabilityUpdateTrackerTicketRequestDTO.TrackerTicket, vulnerabilityID)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, echo.Map{})
}

func (c *Controller) createVulnerabilityTrackerTicket(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "vulnerabilityId", e.Param("vulnerabilityId"))
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilityID, err := strconv.Atoi(e.Param("vulnerabilityId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerability, err := c.vulnerabilityUsecase.GetByVulnerabilityIDAndScanInstanceID(e.Request().Context(),
		vulnerabilityID, scanInstanceID)
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilityCreateTrackerTicketRequestDTO := new(models.VulnerabilityCreateTrackerTicketRequestDTO)
	if err := e.Bind(vulnerabilityCreateTrackerTicketRequestDTO); err != nil {
		return utils.APIError(e, err)
	}
	currentUser := e.Get("user").(*user.User)
	tags := []string{"impulse", e.Param("scanTypeName")}
	issueParameters := secnotify.CreateTicketIssueParameters{
		Queue:            vulnerabilityCreateTrackerTicketRequestDTO.TrackerQueue,
		Summary:          vulnerabilityCreateTrackerTicketRequestDTO.Title,
		Tags:             tags,
		CreatedBy:        currentUser.Login,
		Followers:        vulnerabilityCreateTrackerTicketRequestDTO.Followers,
		Security:         secnotify.Yes,
		SecuritySeverity: secnotify.SeverityTrackerMap[*vulnerability.Severity],
	}
	trackerTicket, err := c.SecNotify.CreateTicket(e.Request().Context(), "impulse_results",
		map[string]interface{}{"ticket_body": vulnerabilityCreateTrackerTicketRequestDTO.Description},
		issueParameters)
	if err != nil {
		return utils.APIError(e, err)
	}
	err = c.vulnerabilityUsecase.UpdateTrackerTicketByID(e.Request().Context(),
		trackerTicket, vulnerabilityID)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, models.VulnerabilityCreateTrackerTicketResponseDTO{TrackerTicket: trackerTicket})
}

func (c *Controller) getCategories(e echo.Context) error {
	simplelog.Info(e.Path(), "organizationId", e.Param("organizationId"), "projectId",
		e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"))

	projectID, err := strconv.Atoi(e.Param("projectId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanTypeName := e.Param("scanTypeName")

	currentScan, err := c.scanUsecase.GetByProjectIDAndScanTypeName(e.Request().Context(), projectID, scanTypeName)
	if err != nil {
		return utils.APIError(e, err)
	}

	categories, err := c.vulnerabilityUsecase.ListVulnerabilityCategories(e.Request().Context(), currentScan.ScanTypeID, currentScan.ID)
	if err != nil {
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, categories)
}

func (c *Controller) getCategoriesForInstance(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"))
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APINotFound(e)
	}
	categories, err := c.vulnerabilityCategoryUsecase.ListByScanInstance(e.Request().Context(), *scanInstance)
	if err != nil {
		return utils.APINotFound(e)
	}

	return utils.APIOk(e, categories)
}

func (c *Controller) getScanInstanceCategoryVulnerabilities(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "categoryId", e.Param("categoryId"),
		"limit", e.QueryParam("limit"), "offset", e.QueryParam("offset"))
	limit, err := strconv.Atoi(e.QueryParam("limit"))
	if err != nil {
		limit = 50
	}
	offset, err := strconv.Atoi(e.QueryParam("offset"))
	if err != nil {
		offset = 0
	}

	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	categoryID, err := strconv.Atoi(e.Param("categoryId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilities, err := c.vulnerabilityUsecase.FetchByScanInstanceIDAndCategoryID(e.Request().Context(),
		scanInstanceID, categoryID, limit, offset)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, vulnerabilities)
}

func (c *Controller) getScanInstanceCategoryInfo(e echo.Context) error {
	simplelog.Info(e.Request().Method+" "+e.Path(), "organizationId", e.Param("organizationId"),
		"projectId", e.Param("projectId"), "scanTypeName", e.Param("scanTypeName"),
		"scanInstanceId", e.Param("scanInstanceId"), "categoryId", e.Param("categoryId"))
	scanInstanceID, err := strconv.Atoi(e.Param("scanInstanceId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	scanInstance, err := c.scanInstanceUsecase.GetByID(e.Request().Context(), scanInstanceID)
	if err != nil {
		return utils.APINotFound(e)
	}
	categoryID, err := strconv.Atoi(e.Param("categoryId"))
	if err != nil {
		return utils.APIError(e, err)
	}
	vulnerabilityCategoryStatistics, err := c.vulnerabilityCategoryUsecase.GetInfoByScanInstanceAndID(e.Request().Context(),
		*scanInstance, categoryID)
	if err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, vulnerabilityCategoryStatistics)
}
