package server

import (
	"context"
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/api/internal/middlewares/auth"
	"a.yandex-team.ru/security/impulse/api/internal/utils"
	"a.yandex-team.ru/security/impulse/api/storage-api/internal/infra"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	Server struct {
		echo        *echo.Echo
		infra       *infra.Infra
		controllers serverControllers
	}
)

func New(infra *infra.Infra) (*Server, error) {
	return &Server{
		echo:        echo.New(),
		infra:       infra,
		controllers: newControllers(infra),
	}, nil
}

func (s *Server) onStart() (err error) {
	return s.infra.Start()
}

func (s *Server) onEnd() (err error) {
	return s.infra.Done()
}

func (s *Server) ListenAndServe(port uint32) error {
	err := s.onStart()
	if err != nil {
		return xerrors.Errorf("failed to start server: %w", err)
	}

	defer func() {
		err := s.onEnd()
		if err != nil {
			simplelog.Error("failed to stop", "err", err)
		}
	}()

	s.echo.HTTPErrorHandler = utils.HTTPErrorHandler

	s.echo.Pre(middleware.RemoveTrailingSlash())

	s.echo.GET("/ping", func(c echo.Context) error {
		return c.String(http.StatusOK, "pong")
	})

	s.echo.GET("/version", func(c echo.Context) error {
		return c.String(http.StatusOK, buildinfo.Info.ProgramVersion)
	})

	authMiddleware := auth.NewAuthMiddleware(s.infra.DB, s.infra.TVM, s.infra.BlackBox, s.infra.CFG.UseAuth)
	api := s.echo.Group("/api/v1", authMiddleware.WithAuth)

	err = s.controllers.VulnController.BuildRoute(api.Group("/storage"))
	if err != nil {
		return err
	}

	err = s.controllers.OrganizationController.BuildRoute(api.Group("/storage"))
	if err != nil {
		return err
	}

	err = s.controllers.ProjectController.BuildRoute(api.Group("/storage"))
	if err != nil {
		return err
	}

	err = s.controllers.TaskController.BuildRoute(api.Group("/storage"))
	if err != nil {
		return err
	}

	err = s.controllers.ScanController.BuildRoute(api.Group("/storage"))
	if err != nil {
		return err
	}

	err = s.controllers.ScanTypeController.BuildRoute(api.Group("/storage"))
	if err != nil {
		return err
	}

	return s.echo.Start(fmt.Sprintf(":%d", port))
}

func (s *Server) Shutdown(ctx context.Context) error {
	return s.echo.Shutdown(ctx)
}
