package codeqldb

import (
	"context"
	"database/sql"
	"time"

	"a.yandex-team.ru/security/impulse/api/repositories/codeqldb"
	"a.yandex-team.ru/security/impulse/models"
)

type codeQLDatabaseUsecase struct {
	codeQLDatabaseRepo codeqldb.Repository
}

func (c codeQLDatabaseUsecase) Get(ctx context.Context, project models.Project, lang string,
	tag string) (*models.CodeQLDatabase, error) {
	return c.codeQLDatabaseRepo.Get(ctx, project.ID, lang, tag)
}

func (c codeQLDatabaseUsecase) Save(ctx context.Context, project models.Project, lang string, tag string,
	mdsURL string, revision string) (*models.CodeQLDatabase, error) {
	codeQLDatabase, err := c.Get(ctx, project, lang, tag)
	if err == sql.ErrNoRows {
		codeQLDatabase = &models.CodeQLDatabase{
			ID:             0,
			OrganizationID: project.OrganizationID,
			ProjectID:      project.ID,
			Language:       lang,
			Tag:            tag,
			MdsURL:         mdsURL,
			ArchiveTime:    time.Now(),
			Revision:       revision,
		}
		return c.codeQLDatabaseRepo.Create(ctx, codeQLDatabase)
	} else if err == nil {
		codeQLDatabase.MdsURL = mdsURL
		codeQLDatabase.Revision = revision
		return codeQLDatabase, c.codeQLDatabaseRepo.Update(ctx, codeQLDatabase)
	} else {
		return nil, err
	}
}

func (c codeQLDatabaseUsecase) ListTags(ctx context.Context, project models.Project,
	lang string) ([]string, error) {
	return c.codeQLDatabaseRepo.ListTags(ctx, project.ID, lang)
}

func (c codeQLDatabaseUsecase) GetByMDS(ctx context.Context, mdsURL string) (*models.CodeQLDatabase, error) {
	return c.codeQLDatabaseRepo.GetByMdsURL(ctx, mdsURL)
}

func (c codeQLDatabaseUsecase) CreateBuildTask(ctx context.Context,
	task *models.CodeQLBuildTask) (*models.CodeQLBuildTask, error) {
	return c.codeQLDatabaseRepo.CreateBuildTask(ctx, task)
}

func (c codeQLDatabaseUsecase) GetOldestBuildTasks(ctx context.Context,
	timestamp int64, count int) ([]*models.CodeQLBuildOldestTask, error) {
	return c.codeQLDatabaseRepo.GetOldestBuildTasks(ctx, timestamp, count)
}

func (c codeQLDatabaseUsecase) Delete(ctx context.Context,
	codeQLDatabase models.CodeQLDatabase) error {
	return c.codeQLDatabaseRepo.Delete(ctx, codeQLDatabase)
}

func NewCodeQLDatabaseUsecase(codeQLDatabaseRepo codeqldb.Repository) Usecase {
	return &codeQLDatabaseUsecase{codeQLDatabaseRepo: codeQLDatabaseRepo}
}
