package idm

import (
	"context"
	"errors"

	"github.com/jackc/pgconn"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/api/repositories/idm"
	"a.yandex-team.ru/security/impulse/models"
)

var ErrRoleAlreadyExists = errors.New("role already exists")

type idmUsecase struct {
	repo idm.Repository
}

func NewIdmUsecase(repo idm.Repository) Usecase {
	return &idmUsecase{repo: repo}
}

func (i idmUsecase) RemoveByLogin(ctx context.Context, login string) error {
	return i.repo.RemoveByLogin(ctx, login)
}

func (i idmUsecase) GetByLogin(ctx context.Context, login string) (*models.IdmUser, error) {
	return i.repo.GetByLogin(ctx, login)
}

func (i idmUsecase) GetOrCreateByLogin(ctx context.Context, login string, isTvmApp bool) (*models.IdmUser, error) {
	user, err := i.repo.GetByLogin(ctx, login)
	switch err {
	case db.ErrNotFound:
		user, err := i.repo.Create(ctx, &models.IdmUser{Login: login, IsTvmApp: isTvmApp})
		return user, err
	case nil:
		return user, nil
	default:
		return nil, err
	}
}

func (i idmUsecase) IsValidRole(role string) bool {
	return !(role != models.VIEW && role != models.TRIAGE && role != models.ADMIN)
}

func (i idmUsecase) AddUserRole(ctx context.Context, role *models.IdmUserRole) error {
	err := i.repo.AddUserRole(ctx, role)
	if err != nil {
		if err.(*pgconn.PgError).ConstraintName == "idmuserrole_pkey" {
			return ErrRoleAlreadyExists
		} else {
			return err
		}
	}
	return nil
}

func (i idmUsecase) RemoveUserRoleByLogin(ctx context.Context, login string, role *models.IdmRoleDTO) error {
	return i.repo.RemoveUserRoleByLogin(ctx, login, role)
}

func (i idmUsecase) ListUserRolesByLogin(ctx context.Context, login string) ([]*models.IdmUserRole, error) {
	return i.repo.ListUserRolesByLogin(ctx, login)
}

func (i idmUsecase) ListRoles(ctx context.Context) ([]*models.IdmUserRole, error) {
	return i.repo.ListRoles(ctx)
}

func (i idmUsecase) ListUsers(ctx context.Context) ([]*models.IdmUser, error) {
	return i.repo.ListUsers(ctx)
}
