package organization

import (
	"context"

	"a.yandex-team.ru/security/impulse/api/repositories/organization"
	"a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	"a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	"a.yandex-team.ru/security/impulse/models"
)

type organizationUsecase struct {
	organizationRepo    organization.Repository
	projectUsecase      project.Usecase
	scanInstanceRepo    scaninstance.Repository
	scanInstanceUsecase _scanInstanceUsecase.Usecase
}

func (o organizationUsecase) DeleteByID(ctx context.Context, id int) error {
	return o.organizationRepo.DeleteByID(ctx, id)
}

func (o organizationUsecase) Create(ctx context.Context, org *models.OrganizationRequestDTO) (models.Organization, error) {
	return o.organizationRepo.Create(ctx, &models.Organization{
		Name:         org.Name,
		Slug:         org.Slug,
		TrackerQueue: org.TrackerQueue,
	})
}

func (o organizationUsecase) Update(ctx context.Context, org *models.Organization) error {
	return o.organizationRepo.Update(ctx, org)
}

func (o organizationUsecase) GetByID(ctx context.Context, id int) (*models.Organization, error) {
	return o.organizationRepo.GetByID(ctx, id)
}

func (o organizationUsecase) GetIDBySlug(ctx context.Context, slug string) (int, error) {
	org, err := o.organizationRepo.GetBySlug(ctx, slug)
	if err != nil {
		return 0, err
	}
	return org.ID, nil
}

func (o organizationUsecase) GetStatisticsByID(ctx context.Context, id int) (*models.OrganizationStatistics, error) {
	organizationStatistics, err := o.organizationRepo.GetSummaryStatisticsByID(ctx, id)
	if err != nil {
		return nil, err
	}
	return organizationStatistics, nil
}

func (o organizationUsecase) List(ctx context.Context) ([]*models.OrganizationInfo, error) {
	return o.organizationRepo.List(ctx)
}

func NewOrganizationUsecase(organizationRepo organization.Repository, projectUsecase project.Usecase,
	scanInstanceRepo scaninstance.Repository, scanInstanceUsecase _scanInstanceUsecase.Usecase) Usecase {
	return &organizationUsecase{
		organizationRepo:    organizationRepo,
		projectUsecase:      projectUsecase,
		scanInstanceRepo:    scanInstanceRepo,
		scanInstanceUsecase: scanInstanceUsecase,
	}
}
