package scan

import (
	"context"
	"database/sql"

	"a.yandex-team.ru/security/impulse/api/internal/db"
	"a.yandex-team.ru/security/impulse/api/repositories/project"
	"a.yandex-team.ru/security/impulse/api/repositories/scan"
	"a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	"a.yandex-team.ru/security/impulse/api/repositories/scantype"
	"a.yandex-team.ru/security/impulse/api/repositories/workflow"
	"a.yandex-team.ru/security/impulse/models"
)

type scanUsecase struct {
	scanRepo         scan.Repository
	projectRepo      project.Repository
	scanTypeRepo     scantype.Repository
	scanInstanceRepo scaninstance.Repository
	workflowRepo     workflow.Repository
}

func NewScanUsecase(scanRepo scan.Repository, projectRepo project.Repository, scanTypeRepo scantype.Repository,
	scanInstanceRepo scaninstance.Repository, workflowRepo workflow.Repository) Usecase {
	return &scanUsecase{
		scanRepo:         scanRepo,
		projectRepo:      projectRepo,
		scanTypeRepo:     scanTypeRepo,
		scanInstanceRepo: scanInstanceRepo,
		workflowRepo:     workflowRepo,
	}
}

func (s scanUsecase) GetByProjectIDAndScanTypeName(ctx context.Context, projectID int, scanTypeName string) (*models.Scan, error) {
	return s.scanRepo.GetByProjectIDAndScanTypeName(ctx, projectID, scanTypeName)
}

func (s scanUsecase) GetInfoByProjectIDAndScanTypeName(ctx context.Context, projectID int, scanTypeName string) (*models.ScanInfo, error) {
	scan, err := s.scanRepo.GetByProjectIDAndScanTypeName(ctx, projectID, scanTypeName)
	if err != nil {
		return nil, err
	}
	scanType, err := s.scanTypeRepo.GetByScanTypeName(ctx, scanTypeName)
	if err != nil {
		return nil, err
	}
	lastScanInstance, err := s.scanRepo.GetLastScanInstance(ctx, scan)
	if err == sql.ErrNoRows {
		scanInfo := models.ScanInfo{
			ID:       scan.ID,
			ScanType: *scanType,
			LastScanInstanceInfo: models.ScanInstanceInfo{
				ScanInstance:           models.ScanInstance{},
				ScanInstanceStatistics: models.ScanInstanceStatistics{},
			},
		}
		return &scanInfo, nil
	}
	if err != nil {
		return nil, err
	}
	lastScanInstanceStatistics, err := s.scanInstanceRepo.GetStatisticsByID(ctx, lastScanInstance.ID)
	if err != nil {
		return nil, err
	}

	scanInfo := models.ScanInfo{
		ID:       scan.ID,
		ScanType: *scanType,
		LastScanInstanceInfo: models.ScanInstanceInfo{
			ScanInstance:           *lastScanInstance,
			ScanInstanceStatistics: *lastScanInstanceStatistics,
		},
	}
	return &scanInfo, nil
}

func (s scanUsecase) GetOrCreateByProjectIDAndScanTypeName(ctx context.Context, projectID int, scanTypeName string) (*models.Scan, error) {
	currentScan, err := s.scanRepo.GetByProjectIDAndScanTypeName(ctx, projectID, scanTypeName)
	if err != nil {
		if err == db.ErrNotFound {
			currentProject, err := s.projectRepo.GetByID(ctx, projectID)
			if err != nil {
				return nil, err
			}
			currentScanType, err := s.scanTypeRepo.GetByScanTypeName(ctx, scanTypeName)
			if err != nil {
				return nil, err
			}
			currentScan, err = s.scanRepo.CreateNew(ctx, currentProject, currentScanType)
			if err != nil {
				return nil, err
			}
			return currentScan, nil
		}
		return nil, err
	}
	return currentScan, nil
}

func (s scanUsecase) CreateScansFromWorkflowID(ctx context.Context, projectID int, workflowID string) ([]*models.Scan, error) {
	workflowScanTypes, err := s.workflowRepo.ListWorkflowScanTypes(ctx, workflowID)
	if err != nil {
		return nil, err
	}

	scans := make([]*models.Scan, len(workflowScanTypes))

	for _, workflow2ScanType := range workflowScanTypes {
		scanType, err := s.scanTypeRepo.GetByID(ctx, workflow2ScanType.ScanTypeID)
		if err != nil {
			return nil, err
		}
		scan, err := s.GetOrCreateByProjectIDAndScanTypeName(ctx, projectID, string(scanType.TypeName))
		if err != nil {
			return nil, err
		}
		scans = append(scans, scan)
	}

	return scans, nil
}

func (s scanUsecase) CreateScansFromAnalysersList(ctx context.Context, projectID int, analysers models.TaskAnalysers) ([]*models.Scan, error) {
	scans := make([]*models.Scan, len(analysers))

	for _, scanTypeName := range analysers {
		scanType, err := s.scanTypeRepo.GetByScanTypeName(ctx, scanTypeName)
		if err != nil {
			return nil, err
		}
		scan, err := s.GetOrCreateByProjectIDAndScanTypeName(ctx, projectID, string(scanType.TypeName))
		if err != nil {
			return nil, err
		}
		scans = append(scans, scan)
	}

	return scans, nil
}

func (s scanUsecase) GetByID(ctx context.Context, id int) (*models.Scan, error) {
	return s.scanRepo.GetByID(ctx, id)
}

func (s scanUsecase) DeleteByID(ctx context.Context, id int) error {
	return s.scanRepo.DeleteByID(ctx, id)
}
