package scaninstance

import (
	"context"
	"database/sql"

	"a.yandex-team.ru/security/impulse/api/repositories/scan"
	"a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	"a.yandex-team.ru/security/impulse/models"
)

type scanInstanceUsecase struct {
	scanRepo         scan.Repository
	scanInstanceRepo scaninstance.Repository
}

func NewScanInstanceStatusUsecase(scanRepo scan.Repository, scanInstanceRepo scaninstance.Repository) Usecase {
	return &scanInstanceUsecase{
		scanRepo:         scanRepo,
		scanInstanceRepo: scanInstanceRepo,
	}
}

func (s scanInstanceUsecase) GetByID(ctx context.Context, id int) (*models.ScanInstance, error) {
	return s.scanInstanceRepo.GetByID(ctx, id)
}

func (s scanInstanceUsecase) Create(ctx context.Context, scanInstance *models.ScanInstance) (*models.ScanInstance, error) {
	createdScanInstance, err := s.scanInstanceRepo.Create(ctx, scanInstance)
	if err != nil {
		return createdScanInstance, err
	}

	err = s.scanInstanceRepo.UpdateLastScanInstance(ctx, createdScanInstance)
	return createdScanInstance, err
}

func (s scanInstanceUsecase) InsertVulnerabilities2ScanInstance(ctx context.Context, scanInstance *models.ScanInstance,
	vulnerabilities []*models.Vulnerability) ([]*models.Vulnerability2ScanInstance, error) {
	return s.scanInstanceRepo.InsertVulnerabilities2ScanInstance(ctx, scanInstance, vulnerabilities)
}

func (s scanInstanceUsecase) ListByTaskID(ctx context.Context, taskID string) ([]*models.ScanInstance, error) {
	return s.scanInstanceRepo.ListByTaskID(ctx, taskID)
}

func (s scanInstanceUsecase) ListByScanID(ctx context.Context, scanID int) ([]*models.ScanInstance, error) {
	return s.scanInstanceRepo.ListByScanID(ctx, scanID)
}

func (s scanInstanceUsecase) GetSummaryStatisticsFromScanInstances(ctx context.Context, scanInstances []*models.ScanInstance) (*models.ScanInstanceStatistics, error) {
	return s.scanInstanceRepo.GetSummaryStatisticsFromScanInstances(ctx, scanInstances)
}

func (s scanInstanceUsecase) GetInfoByID(ctx context.Context, id int) (*models.ScanInstanceInfo, error) {
	scanInstance, err := s.scanInstanceRepo.GetByID(ctx, id)
	if err != nil {
		return nil, err
	}
	scanInstanceStatistics, err := s.scanInstanceRepo.GetStatisticsByID(ctx, id)
	if err != nil {
		return nil, err
	}

	return &models.ScanInstanceInfo{
		ScanInstance:           *scanInstance,
		ScanInstanceStatistics: *scanInstanceStatistics,
	}, nil
}

func (s scanInstanceUsecase) ListLastScanInstancesByProjectID(ctx context.Context, projectID int) ([]*models.ScanInstance, error) {
	scans, err := s.scanRepo.ListByProjectID(ctx, projectID)
	if err != nil {
		return nil, err
	}
	scanInstances := make([]*models.ScanInstance, 0)
	for _, scan := range scans {
		scanInstance, err := s.scanRepo.GetLastScanInstance(ctx, scan)
		if err == sql.ErrNoRows {
			continue
		}
		if err != nil {
			return nil, err
		}
		scanInstances = append(scanInstances, scanInstance)
	}
	return scanInstances, nil
}

func (s scanInstanceUsecase) ListLastScanInstancesByOrganizationID(ctx context.Context,
	organizationID int) ([]*models.ScanInstance, error) {
	return s.scanInstanceRepo.GetLastScanInstancesByOrganizationID(ctx, organizationID)
}
