package task

import (
	"fmt"
	"net/url"
	"regexp"
	"strings"

	"golang.org/x/xerrors"
)

func normalizeRepositoryURL(repoURL string) (string, error) {
	r, err := regexp.Compile(`[\w+]+://`)
	if err != nil {
		return "", err
	}
	loc := r.FindStringIndex(repoURL)
	if loc == nil || loc[0] != 0 {
		repoURL = fmt.Sprintf("ssh://%s", repoURL)
	}
	u, err := url.Parse(repoURL)
	if err != nil {
		return "", err
	}

	switch u.Host {
	case "github.yandex-team.ru", "github.com":
		err = normalizeGitHubURL(u)
	case "git.adfox.ru":
		err = normalizeAdfoxGitLabURL(u)
	case "a.yandex-team.ru", "arcadia.yandex.ru":
		err = normalizeArcadiaURL(u)
	case "bb.yandex-team.ru", "bitbucket.sdc.yandex-team.ru", "bitbucket.browser.yandex-team.ru":
		err = normalizeBitbucketURL(u)
	case "gitlab.edadeal.yandex-team.ru", "git.edadeal.yandex-team.ru":
		err = normalizeEdadelGitLabURL(u)
	default:
		err = xerrors.Errorf("unknown repo: %s", repoURL)
	}

	if err != nil {
		return "", err
	}

	return u.String(), nil
}

func normalizeGitHubURL(u *url.URL) error {
	parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")

	if len(parts) >= 2 {
		u.Path = strings.Join([]string{parts[0], parts[1]}, "/")
	} else {
		return xerrors.Errorf("unknown format of GitHub repo: %s", u.String())
	}

	if !strings.HasSuffix(u.Path, ".git") {
		u.Path += ".git"
	}

	u.Scheme = "ssh"
	u.User = url.User("git")

	return nil
}

func normalizeAdfoxGitLabURL(u *url.URL) error {
	parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")

	if len(parts) >= 2 {
		u.Path = strings.Join([]string{parts[0], parts[1]}, "/")
	} else {
		return xerrors.Errorf("unknown format of Adfox GitLab repo: %s", u.String())
	}

	if !strings.HasSuffix(u.Path, ".git") {
		u.Path += ".git"
	}

	u.Scheme = "http"

	return nil
}

func normalizeEdadelGitLabURL(u *url.URL) error {

	parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")

	if len(parts) >= 2 {
		u.Path = strings.Join([]string{parts[0], parts[1]}, "/")
	} else {
		return xerrors.Errorf("unknown format of GitLab repo: %s", u.String())
	}

	if !strings.HasSuffix(u.Path, ".git") {
		u.Path += ".git"
	}
	u.Host = "git.edadeal.yandex-team.ru"
	u.Scheme = "ssh"
	u.User = url.User("git")

	return nil
}

func normalizeArcadiaURL(u *url.URL) error {
	u.Scheme = "svn+ssh"
	u.Host = "arcadia.yandex.ru"
	if strings.HasPrefix(u.Path, "/arc_vcs/") {
		u.Path = "/arc/trunk/arcadia/" + u.Path[9:]
	}

	return nil
}

func normalizeBitbucketURL(u *url.URL) error {
	parts := strings.Split(strings.TrimLeft(u.Path, "/"), "/")

	switch {
	case len(parts) == 2:
		break
	case len(parts) >= 3 && parts[0] == "scm":
		u.Path = strings.Join([]string{parts[1], parts[2]}, "/")
	case len(parts) >= 4 && parts[0] == "projects" && parts[2] == "repos":
		u.Path = strings.Join([]string{parts[1], parts[3]}, "/")
	case len(parts) >= 4 && parts[0] == "users" && parts[2] == "repos":
		u.Path = strings.Join([]string{fmt.Sprintf("~%s", parts[1]), parts[3]}, "/")
	default:
		return xerrors.Errorf("unknown format of Bitbucket repo: %s", u.String())
	}

	if !strings.HasSuffix(u.Path, ".git") {
		u.Path += ".git"
	}

	u.Scheme = "ssh"

	return nil
}
