package task

import (
	"context"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/security/impulse/api/repositories/task"
	"a.yandex-team.ru/security/impulse/models"
)

type taskUsecase struct {
	repo task.Repository
}

func NewTaskUsecase(repo task.Repository) Usecase {
	return &taskUsecase{
		repo: repo,
	}
}

func (m *taskUsecase) Create(ctx context.Context, t *models.Task) error {
	return m.repo.Create(ctx, t)
}

func (m *taskUsecase) Update(ctx context.Context, t *models.Task) error {
	num, err := m.repo.Update(ctx, t)
	switch {
	case err != nil:
		return err
	case num != 1:
		return xerrors.Errorf("number of updated rows != 1 %v", t.TaskID)
	default:
		return nil
	}
}

func (m *taskUsecase) GetByTaskID(ctx context.Context, taskID string) (*models.Task, error) {
	return m.repo.GetByTaskID(ctx, taskID)
}

func (m *taskUsecase) CheckFinished(ctx context.Context, taskID string) (bool, error) {
	return m.repo.CheckFinished(ctx, taskID)
}

func (m *taskUsecase) GetLastTemplate(ctx context.Context, projectID int) (*models.TaskResponseDTO, error) {
	return m.repo.GetLastTemplate(ctx, projectID)
}

func (m *taskUsecase) List(ctx context.Context, organizationID, projectID, limit, offset int) ([]*models.TaskResponseDTO, error) {
	return m.repo.List(ctx, organizationID, projectID, limit, offset)
}

func (m *taskUsecase) ListByOrganizationID(ctx context.Context, organizationID, limit, offset int) ([]*models.TaskResponseDTO, error) {
	return m.repo.ListByOrganizationID(ctx, organizationID, limit, offset)
}

func (m *taskUsecase) GetTotal(ctx context.Context, organizationID, projectID int) (count int, err error) {
	return m.repo.GetTotal(ctx, organizationID, projectID)
}

func (m *taskUsecase) GetTotalByOrganizationID(ctx context.Context, organizationID int) (count int, err error) {
	return m.repo.GetTotalByOrganizationID(ctx, organizationID)
}

var taskStatusToString = map[models.TaskStatus]string{
	models.Created:                "CREATED",
	models.Pending:                "PENDING",
	models.Running:                "RUNNING",
	models.Failed:                 "FAILED",
	models.Finished:               "FINISHED",
	models.FinishedCallbackFailed: "FINISHED (callback failed)",
}

func (m *taskUsecase) StatusToString(s *models.Task) string {
	return taskStatusToString[s.Status]
}

func (m *taskUsecase) IsValidStatus(status models.TaskStatus) bool {
	if status < models.Created || status > models.Finished {
		return false
	}
	return true
}

func (m *taskUsecase) IsFinalStatus(status models.TaskStatus) bool {
	if status == models.Failed || status == models.Finished || status == models.FinishedCallbackFailed {
		return true
	}
	return false
}

func (m *taskUsecase) NormalizeTaskParameters(params models.TaskParameters) (models.TaskParameters, error) {
	newParams := make(models.TaskParameters)
	for key, value := range params {
		switch key {
		case "repositories":
			repositories, ok := value.([]interface{})
			if !ok {
				return nil, xerrors.Errorf("unexpected repositories type")
			}
			newRepositories := make([]models.Repository, len(repositories))
			for i := range repositories {
				repository, ok := repositories[i].(map[string]interface{})
				if !ok {
					return nil, xerrors.Errorf("unexpected repositories type")
				}
				repositoryURL, ok := repository["url"].(string)
				if !ok {
					return nil, xerrors.Errorf("unexpected repository url type")
				}
				normalizedRepoURL, err := normalizeRepositoryURL(repositoryURL)
				if err != nil {
					return nil, err
				}
				newRepositories[i].URL = normalizedRepoURL
				if repositoryBranch, ok := repository["branch"].(string); ok {
					newRepositories[i].Branch = repositoryBranch
				}
				if repositoryRevision, ok := repository["revision"].(string); ok {
					newRepositories[i].Revision = repositoryRevision
				}
				if repositoryPath, ok := repository["path"].(string); ok {
					newRepositories[i].Path = repositoryPath
				}
			}
			newParams[key] = newRepositories
		case "include_patterns", "exclude_patterns":
			patterns, ok := value.([]interface{})
			if !ok {
				patterns = make([]interface{}, 0)
			}
			newParams[key] = patterns
		default:
			newParams[key] = value
		}
	}

	return newParams, nil
}
