package vulnerability

import (
	"context"
	"time"

	"a.yandex-team.ru/security/impulse/api/repositories/vulnerability"
	"a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitytotalstatistics"
	"a.yandex-team.ru/security/impulse/models"
)

type VulnerabilityUsecase struct {
	repo                         vulnerability.Repository
	vulnerabilityTotalStatistics vulnerabilitytotalstatistics.Usecase
}

func NewVulnerabilityUsecase(repo vulnerability.Repository,
	vulnerabilityTotalStatistics vulnerabilitytotalstatistics.Usecase) Usecase {
	return &VulnerabilityUsecase{
		repo:                         repo,
		vulnerabilityTotalStatistics: vulnerabilityTotalStatistics,
	}
}

func (v VulnerabilityUsecase) FetchByScanID(ctx context.Context, scanID int) ([]*models.Vulnerability, error) {
	vulnerabilities, err := v.repo.FetchByScanID(ctx, scanID)
	if err != nil {
		return nil, err
	}
	return vulnerabilities, nil
}

func (v VulnerabilityUsecase) UpdateByScanID(ctx context.Context, scanID int, vulnerabilities []*models.Vulnerability) error {
	return v.repo.UpdateByScanID(ctx, scanID, vulnerabilities)
}

func (v VulnerabilityUsecase) InsertByScanIDAndLastUpdateToken(ctx context.Context, scanID int, lastUpdateToken string,
	vulnerabilities []*models.Vulnerability) ([]*models.Vulnerability, error) {
	return v.repo.InsertByScanIDAndLastUpdateToken(ctx, scanID, lastUpdateToken, vulnerabilities)
}

func (v VulnerabilityUsecase) UpdateFromScanInstanceVulnerabilities(ctx context.Context, scanInstanceID int) error {
	return v.repo.UpdateFromScanInstanceVulnerabilities(ctx, scanInstanceID)
}

func (v VulnerabilityUsecase) FetchByScanInstanceID(ctx context.Context, scanInstanceID int, limit int, offset int) ([]*models.Vulnerability, error) {
	return v.repo.FetchByScanInstanceID(ctx, scanInstanceID, limit, offset)
}

func (v VulnerabilityUsecase) GetByVulnerabilityIDAndScanInstanceID(ctx context.Context, vulnerabilityID int,
	scanInstanceID int) (*models.Vulnerability, error) {
	return v.repo.GetByVulnerabilityIDAndScanInstanceID(ctx, vulnerabilityID, scanInstanceID)
}

func (v VulnerabilityUsecase) GetByVulnerabilityID(ctx context.Context, vulnerabilityID int) (*models.Vulnerability, error) {
	return v.repo.GetByVulnerabilityID(ctx, vulnerabilityID)
}

func (v VulnerabilityUsecase) ListVulnerabilityCategories(ctx context.Context, scanTypeID, scanID int) ([]*models.VulnerabilityCategoryResponseDTO, error) {
	return v.repo.ListVulnerabilityCategories(ctx, scanTypeID, scanID)
}

func (v VulnerabilityUsecase) UpdateStatusByID(ctx context.Context, status models.StatusType, id int) error {
	vulnerabilityIDs := []int{id}
	err := v.repo.UpdateStatusByIDs(ctx, status, vulnerabilityIDs)
	return err
}

func (v VulnerabilityUsecase) BatchUpdateStatus(ctx context.Context, status models.StatusType, IDs []int) error {
	err := v.repo.UpdateStatusByIDs(ctx, status, IDs)
	return err
}

func (v VulnerabilityUsecase) UpdateTrackerTicketByID(ctx context.Context, trackerTicket string, id int) error {
	return v.repo.UpdateTrackerTicketByID(ctx, trackerTicket, id)
}

func (v VulnerabilityUsecase) FetchByScanInstanceIDAndCategoryID(ctx context.Context, scanInstanceID int, categoryID int,
	limit int, offset int) ([]*models.Vulnerability, error) {
	return v.repo.FetchByScanInstanceIDAndCategoryID(ctx, scanInstanceID, categoryID, limit, offset)
}

func (v VulnerabilityUsecase) FetchByScanInstances(ctx context.Context, scanInstances []*models.ScanInstance,
	limit int, offset int, filterMap map[string]interface{}) ([]*models.Vulnerability, error) {
	return v.repo.FetchByScanInstances(ctx, scanInstances, limit, offset, filterMap)
}

func (v VulnerabilityUsecase) FetchLatestByOrganizationID(ctx context.Context, organizationID int,
	limit int, offset int, filterMap map[string]interface{}) ([]*models.Vulnerability, error) {
	filterAndArray := []interface{}{
		map[string]interface{}{
			"eq": map[string]interface{}{
				"project.organization_id": organizationID,
			},
		},
	}
	if len(filterMap) != 0 {
		filterAndArray = append(filterAndArray, filterMap)
	}
	filterMap = map[string]interface{}{
		"and": filterAndArray,
	}
	return v.repo.FetchLatest(ctx, limit, offset, filterMap)
}

func (v VulnerabilityUsecase) FetchLatestByProjectIDWithoutTicket(ctx context.Context, projectID int, since time.Time,
	limit int, offset int, filterMap map[string]interface{}) ([]*models.Vulnerability, error) {
	filterAndArray := []interface{}{
		map[string]interface{}{
			"eq": map[string]interface{}{
				"project.id": projectID,
			},
		},
		map[string]interface{}{
			"gt": map[string]interface{}{
				"v.first_found_at": since.Format("2006-01-02 15:04:05"),
			},
		},
		map[string]interface{}{
			"eq": map[string]interface{}{
				"v.tracker_ticket": "",
			},
		},
	}
	if len(filterMap) != 0 {
		filterAndArray = append(filterAndArray, filterMap)
	}
	filterMap = map[string]interface{}{
		"and": filterAndArray,
	}
	return v.repo.FetchLatest(ctx, limit, offset, filterMap)
}
