package workflow

import (
	"context"

	"golang.org/x/xerrors"

	"a.yandex-team.ru/security/impulse/api/repositories/workflow"
	"a.yandex-team.ru/security/impulse/models"
)

type workflowUsecase struct {
	repo workflow.Repository
}

func NewWorkflowUsecase(repo workflow.Repository) Usecase {
	return &workflowUsecase{
		repo: repo,
	}
}

func (m *workflowUsecase) Update(ctx context.Context, w *models.Workflow, scanTypeIds []int) error {
	num, err := m.repo.Update(ctx, w)
	if err != nil {
		return err
	} else if num != 1 {
		return xerrors.Errorf("number of updated rows != 1 %v", w.ID)
	}
	return m.repo.UpdateWorkflowScanTypes(ctx, w.ID, scanTypeIds)
}

func (m *workflowUsecase) Get(ctx context.Context, workflowID string) (*models.WorkflowResponseDTO, error) {
	workflow, err := m.repo.Get(ctx, workflowID)
	if err != nil {
		return nil, err
	}

	scanTypes, err := m.repo.ListWorkflowScanTypeTitles(ctx, workflowID)
	if err != nil {
		return nil, err
	}

	result := models.WorkflowResponseDTO{
		Workflow:  *workflow,
		ScanTypes: []string{},
	}
	for _, scanType := range scanTypes {
		result.ScanTypes = append(result.ScanTypes, scanType.ScanTypeTitle)
	}

	return &result, nil
}

func (m *workflowUsecase) List(ctx context.Context) ([]*models.WorkflowResponseDTO, error) {
	workflows, err := m.repo.ListWorkflows(ctx)
	if err != nil {
		return nil, err
	}
	scanTypes, err := m.repo.ListWorkflowScanTypeTitles(ctx, "")
	if err != nil {
		return nil, err
	}

	scanTypesMap := map[string][]string{}
	for _, scanType := range scanTypes {
		scanTypesMap[scanType.WorkflowID] = append(scanTypesMap[scanType.WorkflowID], scanType.ScanTypeTitle)
	}

	result := make([]*models.WorkflowResponseDTO, 0)
	for _, workflow := range workflows {
		workflowResponse := models.WorkflowResponseDTO{
			Workflow:  *workflow,
			ScanTypes: scanTypesMap[workflow.ID],
		}
		result = append(result, &workflowResponse)
	}

	return result, nil
}
