package insights

import (
	"strings"
	"time"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_taskRepo "a.yandex-team.ru/security/impulse/api/repositories/task"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_taskUsecase "a.yandex-team.ru/security/impulse/api/usecases/task"
	"a.yandex-team.ru/security/impulse/api/webhook/internal/infra"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

type (
	Controller struct {
		*infra.Infra

		projectUsecase _projectUsecase.Usecase
		taskUsecase    _taskUsecase.Usecase
	}
)

func (c *Controller) BuildRoute(g *echo.Group) error {

	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	taskRepository := _taskRepo.NewTaskRepository(c.DB)

	scanInstanceUsecase := _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	c.taskUsecase = _taskUsecase.NewTaskUsecase(taskRepository)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, scanInstanceUsecase)

	g.POST("/insights/export", c.insightsExport)
	return nil
}

func (c *Controller) insightsExport(e echo.Context) error {
	r := new(models.WebhookReport)
	simplelog.Info(e.Request().Method + " " + e.Path())
	if err := e.Bind(r); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}

	// Extracting paths from task parameters
	revision := ""
	pathsMap := make(map[string]bool)
	for _, scanInstance := range r.Instances {
		if scanInstance.CommitHash != "" {
			revision = scanInstance.CommitHash
		}
		task, err := c.taskUsecase.GetByTaskID(e.Request().Context(), scanInstance.TaskID)
		if err != nil {
			return utils.APIError(e, err)
		}
		if repositories, ok := task.Parameters["repositories"].([]interface{}); ok {
			for _, repository := range repositories {
				if repo, ok := repository.(map[string]interface{}); ok {
					if repoURL, ok := repo["url"].(string); ok {
						repoPath := strings.TrimPrefix(repoURL, "https://a.yandex-team.ru/arc/trunk/arcadia")
						pathsMap[repoPath] = true
					}
				}
			}
		}
	}
	paths := make([]string, 0)
	for path := range pathsMap {
		paths = append(paths, path)
	}

	// Get project statistics
	vulnerabilityStatistics, err := c.projectUsecase.GetStatisticsByID(e.Request().Context(), r.ProjectID)
	if err != nil {
		return utils.APIError(e, err)
	}
	// Get start timestamp
	startTimestamp := ""
	if len(r.Instances) > 0 {
		startTimestampMoscow := r.Instances[0].StartTime
		startTimestamp = startTimestampMoscow.Add(time.Hour * -3).UTC().Format(time.RFC3339Nano)
	}
	// Fill Metrics
	metrics := make([]models.MetricForInsights, 0)
	metricsData := map[string]int{
		"impulse_count_blocker_issues":  vulnerabilityStatistics.BlockerNotFalseVulnerabilitiesCount,
		"impulse_count_critical_issues": vulnerabilityStatistics.CriticalNotFalseVulnerabilitiesCount,
		"impulse_count_medium_issues":   vulnerabilityStatistics.MediumNotFalseVulnerabilitiesCount,
		"impulse_count_low_issues":      vulnerabilityStatistics.LowNotFalseVulnerabilitiesCount,
		"impulse_count_info_issues":     vulnerabilityStatistics.InfoNotFalseVulnerabilitiesCount,
	}
	for name, value := range metricsData {
		metrics = append(metrics, models.MetricForInsights{
			Category: "security",
			Name:     name,
			Paths:    paths,
			Data: []models.MetricData{{
				Timestamp:  startTimestamp,
				CommitHash: revision,
				Value:      value,
			}},
		})
	}
	if err = c.Insights.SendMetrics(e.Request().Context(), metrics); err != nil {
		return utils.APIError(e, err)
	}
	return utils.APIOk(e, echo.Map{})
}
