package secnotify

import (
	"bytes"
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"
	"golang.org/x/xerrors"

	"a.yandex-team.ru/security/impulse/api/internal/utils"
	_projectRepository "a.yandex-team.ru/security/impulse/api/repositories/project"
	_scanRepository "a.yandex-team.ru/security/impulse/api/repositories/scan"
	_scanInstanceRepository "a.yandex-team.ru/security/impulse/api/repositories/scaninstance"
	_scanTypeRepository "a.yandex-team.ru/security/impulse/api/repositories/scantype"
	_vulnerabilityRepository "a.yandex-team.ru/security/impulse/api/repositories/vulnerability"
	_vulnerabilityTotalStatisticsRepo "a.yandex-team.ru/security/impulse/api/repositories/vulnerabilitytotalstatistics"
	_projectUsecase "a.yandex-team.ru/security/impulse/api/usecases/project"
	_scanInstanceUsecase "a.yandex-team.ru/security/impulse/api/usecases/scaninstance"
	_scanTypeUsecase "a.yandex-team.ru/security/impulse/api/usecases/scantype"
	_vulnerabilityUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerability"
	_vulnerabilityTotalStatisticsUsecase "a.yandex-team.ru/security/impulse/api/usecases/vulnerabilitytotalstatistics"
	"a.yandex-team.ru/security/impulse/api/webhook/internal/infra"
	"a.yandex-team.ru/security/impulse/models"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const Upstream = "https://secnotify.sec.yandex-team.ru"
const TvmID = 2017559

type (
	Template struct {
		TemplateID      string          `json:"template_id"`
		IssueID         string          `json:"issue_id,omitempty"`
		IssueParameters IssueParameters `json:"issue_parameters,omitempty"`
		Parameters      interface{}     `json:"template_parameters"`
	}
	IssueParameters struct {
		Queue      string   `json:"queue,omitempty"`
		Summary    string   `json:"summary,omitempty"`
		Followers  []string `json:"followers,omitempty"`
		Assignee   string   `json:"assignee,omitempty"`
		Security   string   `json:"security,omitempty"`
		Tags       []string `json:"tags,omitempty"`
		Components int      `json:"components,omitempty"`
	}

	Controller struct {
		*infra.Infra

		Client               *http.Client
		projectUsecase       _projectUsecase.Usecase
		vulnerabilityUsecase _vulnerabilityUsecase.Usecase
		scanInstanceUsecase  _scanInstanceUsecase.Usecase
		scanTypeUsecase      _scanTypeUsecase.Usecase
	}
)

func (c *Controller) BuildRoute(g *echo.Group) error {
	c.Client = &http.Client{}

	projectRepository := _projectRepository.NewProjectRepository(c.DB)
	scanRepository := _scanRepository.NewScanRepository(c.DB)
	scanTypeRepository := _scanTypeRepository.NewScanTypeRepository(c.DB)
	scanInstanceRepository := _scanInstanceRepository.NewScanInstanceRepository(c.DB)
	vulnerabilityRepository := _vulnerabilityRepository.NewVulnerabilityRepository(c.DB)

	c.scanInstanceUsecase = _scanInstanceUsecase.NewScanInstanceStatusUsecase(scanRepository, scanInstanceRepository)
	c.scanTypeUsecase = _scanTypeUsecase.NewScanTypeUsecase(scanTypeRepository)
	c.projectUsecase = _projectUsecase.NewProjectUsecase(projectRepository, scanInstanceRepository, c.scanInstanceUsecase)

	vulnerabilityTotalStatisticsRepo := _vulnerabilityTotalStatisticsRepo.NewVulnerabilityTotalStatisticsRepository(c.DB)
	vulnerabilityTotalStatisticsUsecase := _vulnerabilityTotalStatisticsUsecase.NewVulnerabilityTotalStatisticsUsecase(vulnerabilityTotalStatisticsRepo)
	c.vulnerabilityUsecase = _vulnerabilityUsecase.NewVulnerabilityUsecase(vulnerabilityRepository,
		vulnerabilityTotalStatisticsUsecase)

	g.POST("/secnotify/startrek/:id", c.startrek)

	return nil
}

func (c *Controller) startrek(e echo.Context) error {
	id := e.Param("id")
	templateID := e.QueryParam("template_id")
	simplelog.Info(e.Request().Method+" "+e.Path(), "id", id, "template_id", templateID)
	r := new(models.WebhookReport)
	if err := e.Bind(r); err != nil {
		return utils.APIError(e, echo.ErrBadRequest)
	}
	simplelog.Info(e.Request().Method+" "+e.Path(), "report", fmt.Sprintf("%#v", r))

	var data []byte
	var err error
	secnotifyURL := fmt.Sprintf("%s/%s", Upstream, "tracker/comment")
	switch templateID {
	case "":
		fallthrough
	case SecauditScanV2:
		data, err = c.buildSecauditReport(e.Request().Context(), id, r)
	case YCloudSastReport:
		secnotifyURL = fmt.Sprintf("%s/%s", Upstream, "tracker/issue")
		data, err = c.buildVulnerabilityReport(e.Request().Context(), id, r)
	case GenericDependenciesAlert:
		fallthrough
	case YCloudPCIDSSDependenciesAlert:
		secnotifyURL = fmt.Sprintf("%s/%s", Upstream, "tracker/issue")
		data, err = c.buildYCloudPCIDSSDependenciesReport(e.Request().Context(), templateID, id, r)
	default:
		err = xerrors.Errorf("Unknown template_id %s", templateID)
	}
	if err != nil {
		simplelog.Info(e.Request().Method+" "+e.Path(), "err", err)
		return utils.APIError(e, err)
	}
	if data == nil {
		return utils.APIOk(e, echo.Map{})
	}

	simplelog.Info(e.Request().Method+" "+e.Path(), "template", string(data))
	req, _ := http.NewRequest("POST", secnotifyURL, bytes.NewBuffer(data))
	req.Header.Add("Content-Type", "application/json; charset=utf-8")

	ticket, err := c.Infra.TVM.GetServiceTicketForID(e.Request().Context(), TvmID)
	if err != nil {
		err = xerrors.Errorf("failed to receive TVM ticket for %d: %w", TvmID, err)
		return utils.APIError(e, err)
	}
	req.Header.Add("X-Ya-Service-Ticket", ticket)
	if c.Infra.CFG.SecnotifyToken != "" {
		req.Header.Add("Authorization", c.Infra.CFG.SecnotifyToken)
	}

	resp, err := c.Client.Do(req)
	if err != nil {
		return utils.APIError(e, err)
	}
	defer func() { _ = resp.Body.Close() }()
	if resp.StatusCode != http.StatusOK {
		err = xerrors.Errorf("Callback wrong status code %d", resp.StatusCode)
		return utils.APIError(e, err)
	}

	return utils.APIOk(e, echo.Map{})
}
