package secnotify

import (
	"context"
	"encoding/json"
	"fmt"
	"math/rand"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/impulse/models"
)

const YCloudPCIDSSDependenciesAlert = "ycloud_pcidss_dependencies_alert"
const GenericDependenciesAlert = "generic_dependencies_alert"

type (
	YCloudDependenciesTemplateParameters struct {
		TotalNotFalseVulnerabilitiesCount    int    `json:"total_count"`
		BlockerNotFalseVulnerabilitiesCount  int    `json:"blocker_count"`
		CriticalNotFalseVulnerabilitiesCount int    `json:"critical_count"`
		MediumNotFalseVulnerabilitiesCount   int    `json:"medium_count"`
		LowNotFalseVulnerabilitiesCount      int    `json:"low_count"`
		InfoNotFalseVulnerabilitiesCount     int    `json:"info_count"`
		ReportURL                            string `json:"report_url"`
	}
)

func (c *Controller) buildYCloudPCIDSSDependenciesReport(ctx context.Context, templateID, queue string, report *models.WebhookReport) ([]byte, error) {
	project, err := c.projectUsecase.GetByID(ctx, report.ProjectID)
	if err != nil {
		return nil, err
	}

	scanInstanceID := 0
	followersSet := make(map[string]bool)
	for _, scanInstance := range report.Instances {
		if scanInstance.ScanTypeName == string(models.YADI) {
			scanInstanceID = scanInstance.ID
			filterAndArray := []interface{}{
				map[string]interface{}{
					"eq": map[string]interface{}{
						"v.status": models.NotReviewed,
					},
				},
				map[string]interface{}{
					"gt": map[string]interface{}{
						"v.first_found_at": time.Now().Format("2006-01-02"),
					},
				},
			}
			filterMap := map[string]interface{}{
				"and": filterAndArray,
			}
			vulnerabilities, err := c.vulnerabilityUsecase.FetchByScanInstances(ctx, []*models.ScanInstance{scanInstance}, 100, 0, filterMap)
			if err != nil {
				return nil, err
			}
			if len(vulnerabilities) == 0 {
				return nil, nil
			}
			for _, vulnerability := range vulnerabilities {
				owners, ok := vulnerability.DisplayProperties["owners"].([]interface{})
				if !ok {
					continue
				}
				for _, owner := range owners {
					followersSet[owner.(string)] = true
				}
			}
			break
		}
	}

	if scanInstanceID == 0 {
		err = xerrors.Errorf("No yadi report")
		return nil, err
	}

	info, err := c.scanInstanceUsecase.GetInfoByID(ctx, scanInstanceID)
	if err != nil {
		return nil, err
	}

	var followers []string
	for follower := range followersSet {
		followers = append(followers, follower)
	}

	assignee := ""
	rand.Seed(time.Now().UnixNano())
	if len(followers) > 0 {
		assignee = followers[rand.Intn(len(followers))]
	} else if project.ABCServiceID != 0 {
		followers, err = c.ABC.GetMembers(project.ABCServiceID, false, "development")
		if err == nil && len(followers) > 0 {
			assignee = followers[rand.Intn(len(followers))]
		}
	}

	params := IssueParameters{
		Queue:     queue,
		Summary:   fmt.Sprintf("Dependencies alert on %s, %s", project.Name, time.Now().Format("02.01.2006")),
		Followers: followers,
		Assignee:  assignee,
		Security:  "Yes",
		Tags:      []string{"impulse", "yadi_scan"},
	}
	if templateID == YCloudPCIDSSDependenciesAlert {
		params.Components = 51425
		params.Summary = fmt.Sprintf("PCI DSS dependencies alert on %s, %s", project.Name, time.Now().Format("02.01.2006"))
	} else if templateID == GenericDependenciesAlert {
		params.Summary = fmt.Sprintf("Vulnerable dependencies in %s, %s", project.Name, time.Now().Format("02.01.2006"))
	}

	reportURL := fmt.Sprintf("https://impulse.sec.yandex-team.ru/organizations/%d/projects/%d/scans/yadi_scan/instances/%d",
		report.OrganizationID, report.ProjectID, scanInstanceID)

	template := &Template{
		TemplateID:      templateID,
		IssueParameters: params,
		Parameters: YCloudDependenciesTemplateParameters{
			TotalNotFalseVulnerabilitiesCount:    info.ScanInstanceStatistics.TotalNotFalseVulnerabilitiesCount,
			BlockerNotFalseVulnerabilitiesCount:  info.ScanInstanceStatistics.BlockerNotFalseVulnerabilitiesCount,
			CriticalNotFalseVulnerabilitiesCount: info.ScanInstanceStatistics.CriticalNotFalseVulnerabilitiesCount,
			MediumNotFalseVulnerabilitiesCount:   info.ScanInstanceStatistics.MediumNotFalseVulnerabilitiesCount,
			LowNotFalseVulnerabilitiesCount:      info.ScanInstanceStatistics.LowNotFalseVulnerabilitiesCount,
			InfoNotFalseVulnerabilitiesCount:     info.ScanInstanceStatistics.InfoNotFalseVulnerabilitiesCount,
			ReportURL:                            reportURL,
		},
	}

	data, _ := json.Marshal(template)

	return data, nil
}
